import { motion } from 'motion/react';
import { useEffect, useState } from 'react';

interface Coin {
  id: number;
  x: number;
  delay: number;
  duration: number;
}

export function FloatingCoins() {
  const [coins] = useState<Coin[]>(() =>
    Array.from({ length: 8 }, (_, i) => ({
      id: i,
      x: Math.random() * 100,
      delay: Math.random() * 2,
      duration: 3 + Math.random() * 2,
    }))
  );

  return (
    <div className="absolute inset-0 overflow-hidden pointer-events-none">
      {coins.map((coin) => (
        <motion.div
          key={coin.id}
          className="absolute"
          style={{
            left: `${coin.x}%`,
            top: '100%',
          }}
          animate={{
            y: [0, -1000],
            rotate: [0, 360],
            opacity: [0, 1, 1, 0],
          }}
          transition={{
            duration: coin.duration,
            delay: coin.delay,
            repeat: Infinity,
            ease: 'linear',
          }}
        >
          <div className="w-12 h-12 rounded-full bg-gradient-to-br from-yellow-400 to-amber-600 shadow-lg flex items-center justify-center">
            <span className="text-amber-900 font-bold text-lg">$</span>
          </div>
        </motion.div>
      ))}
    </div>
  );
}

export function CasinoRoulette() {
  return (
    <motion.div
      className="relative w-64 h-64"
      animate={{ rotate: 360 }}
      transition={{ duration: 8, repeat: Infinity, ease: 'linear' }}
    >
      <div className="absolute inset-0 rounded-full bg-gradient-to-br from-red-600 via-black to-red-800 shadow-2xl">
        <div className="absolute inset-4 rounded-full bg-gradient-to-br from-amber-500 to-amber-700">
          <div className="absolute inset-4 rounded-full bg-gray-900">
            {Array.from({ length: 12 }).map((_, i) => (
              <div
                key={i}
                className="absolute w-full h-full"
                style={{
                  transform: `rotate(${i * 30}deg)`,
                }}
              >
                <div className={`w-6 h-8 mx-auto ${i % 2 === 0 ? 'bg-red-600' : 'bg-black'} rounded-t`} />
              </div>
            ))}
            <div className="absolute inset-0 flex items-center justify-center">
              <div className="w-8 h-8 rounded-full bg-gradient-to-br from-amber-300 to-amber-500 shadow-lg" />
            </div>
          </div>
        </div>
      </div>
    </motion.div>
  );
}

export function SportsTrophy() {
  return (
    <motion.div
      className="relative w-48 h-56"
      animate={{
        y: [0, -20, 0],
        rotateY: [0, 10, 0, -10, 0],
      }}
      transition={{
        duration: 4,
        repeat: Infinity,
        ease: 'easeInOut',
      }}
    >
      {/* Trophy Cup */}
      <div className="absolute bottom-0 left-1/2 -translate-x-1/2 w-32 h-40">
        <div className="absolute bottom-0 w-full h-8 bg-gradient-to-b from-amber-600 to-amber-800 rounded" />
        <div className="absolute bottom-8 w-full h-4 bg-gradient-to-b from-amber-700 to-amber-900" />
        <div className="absolute bottom-12 left-1/2 -translate-x-1/2 w-24 h-28 bg-gradient-to-br from-yellow-400 via-amber-500 to-amber-600 rounded-t-full shadow-2xl">
          <div className="absolute top-4 left-1/2 -translate-x-1/2 w-16 h-16 bg-gradient-to-br from-amber-300 to-amber-500 rounded-full flex items-center justify-center">
            <span className="text-amber-900 font-bold text-2xl">1</span>
          </div>
        </div>
        {/* Handles */}
        <div className="absolute bottom-16 -left-4 w-6 h-12 bg-gradient-to-br from-amber-500 to-amber-700 rounded-l-full" />
        <div className="absolute bottom-16 -right-4 w-6 h-12 bg-gradient-to-br from-amber-500 to-amber-700 rounded-r-full" />
      </div>
      {/* Sparkles */}
      <motion.div
        className="absolute top-0 left-0 w-3 h-3 bg-yellow-400 rounded-full"
        animate={{
          scale: [0, 1, 0],
          opacity: [0, 1, 0],
        }}
        transition={{
          duration: 2,
          repeat: Infinity,
          delay: 0,
        }}
      />
      <motion.div
        className="absolute top-4 right-4 w-2 h-2 bg-amber-300 rounded-full"
        animate={{
          scale: [0, 1, 0],
          opacity: [0, 1, 0],
        }}
        transition={{
          duration: 2,
          repeat: Infinity,
          delay: 0.5,
        }}
      />
    </motion.div>
  );
}

export function FootballAnimation() {
  return (
    <motion.div
      className="relative w-32 h-32"
      animate={{
        rotate: [0, 360],
        y: [0, -30, 0],
      }}
      transition={{
        rotate: { duration: 2, repeat: Infinity, ease: 'linear' },
        y: { duration: 1.5, repeat: Infinity, ease: 'easeInOut' },
      }}
    >
      <div className="w-full h-full rounded-full bg-gradient-to-br from-white via-gray-100 to-gray-300 shadow-2xl relative overflow-hidden">
        {/* Pentagon pattern */}
        <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-12 h-12 bg-gray-800 transform rotate-45" style={{ clipPath: 'polygon(50% 0%, 100% 38%, 82% 100%, 18% 100%, 0% 38%)' }} />
        {Array.from({ length: 6 }).map((_, i) => (
          <div
            key={i}
            className="absolute w-8 h-10 bg-gray-800"
            style={{
              clipPath: 'polygon(50% 0%, 100% 38%, 82% 100%, 18% 100%, 0% 38%)',
              left: '50%',
              top: '50%',
              transform: `translate(-50%, -50%) rotate(${i * 60}deg) translateY(-35px)`,
            }}
          />
        ))}
      </div>
    </motion.div>
  );
}

export function ChartAnimation() {
  return (
    <motion.div className="relative w-48 h-32 flex items-end justify-around gap-2">
      {[40, 70, 50, 90, 60].map((height, i) => (
        <motion.div
          key={i}
          className="w-8 bg-gradient-to-t from-purple-600 to-purple-400 rounded-t"
          initial={{ height: 0 }}
          animate={{ height: `${height}%` }}
          transition={{
            duration: 1,
            delay: i * 0.1,
            repeat: Infinity,
            repeatType: 'reverse',
            repeatDelay: 1,
          }}
        />
      ))}
    </motion.div>
  );
}
