import { ReactNode } from 'react';
import { Link } from 'react-router';

interface ButtonProps {
  children: ReactNode;
  onClick?: () => void;
  type?: 'button' | 'submit' | 'reset';
  variant?: 'primary' | 'secondary' | 'outline' | 'ghost';
  size?: 'sm' | 'md' | 'lg';
  disabled?: boolean;
  className?: string;
  to?: string;
}

export function BrandButton({
  children,
  onClick,
  type = 'button',
  variant = 'primary',
  size = 'md',
  disabled = false,
  className = '',
  to,
}: ButtonProps) {
  const baseStyles = 'font-semibold transition-all rounded-xl inline-flex items-center justify-center gap-2';
  
  const variantStyles = {
    primary: 'bg-gradient-to-r from-[#450693] to-[#8C00FF] hover:from-[#8C00FF] hover:to-[#FF3F7F] text-white shadow-lg',
    secondary: 'bg-gray-700 hover:bg-gray-600 text-white',
    outline: 'border-2 border-[#8C00FF] text-[#8C00FF] hover:bg-[#8C00FF] hover:text-white',
    ghost: 'text-[#8C00FF] hover:bg-[#8C00FF]/10',
  };
  
  const sizeStyles = {
    sm: 'px-4 py-2 text-sm',
    md: 'px-6 py-3 text-base',
    lg: 'px-8 py-4 text-lg',
  };
  
  const disabledStyles = disabled ? 'opacity-50 cursor-not-allowed' : '';
  
  const finalClassName = `${baseStyles} ${variantStyles[variant]} ${sizeStyles[size]} ${disabledStyles} ${className}`;
  
  if (to) {
    return (
      <Link to={to} className={finalClassName}>
        {children}
      </Link>
    );
  }
  
  return (
    <button
      type={type}
      onClick={onClick}
      disabled={disabled}
      className={finalClassName}
    >
      {children}
    </button>
  );
}

// Logo component with brand colors
export function BrandLogo({ size = 'md' }: { size?: 'sm' | 'md' | 'lg' }) {
  const sizeMap = {
    sm: 'w-8 h-8 text-lg',
    md: 'w-10 h-10 text-xl',
    lg: 'w-16 h-16 text-3xl',
  };
  
  return (
    <div 
      className={`${sizeMap[size]} rounded-lg flex items-center justify-center`}
      style={{ background: 'linear-gradient(135deg, #450693 0%, #8C00FF 100%)' }}
    >
      <span className="text-white font-bold">7★</span>
    </div>
  );
}

// Progress bar with brand gradient
export function BrandProgressBar({ progress }: { progress: number }) {
  return (
    <div className="w-full h-2 bg-gray-800 rounded-full overflow-hidden">
      <div
        className="h-full transition-all duration-300"
        style={{
          width: `${progress}%`,
          background: 'linear-gradient(90deg, #450693 0%, #8C00FF 50%, #FF3F7F 100%)',
        }}
      />
    </div>
  );
}

// Badge with brand colors
export function BrandBadge({ 
  children, 
  variant = 'purple' 
}: { 
  children: ReactNode; 
  variant?: 'purple' | 'pink' | 'gold' 
}) {
  const variants = {
    purple: 'bg-[#8C00FF]/20 text-[#8C00FF] border-[#8C00FF]/30',
    pink: 'bg-[#FF3F7F]/20 text-[#FF3F7F] border-[#FF3F7F]/30',
    gold: 'bg-[#FFC400]/20 text-[#FFC400] border-[#FFC400]/30',
  };
  
  return (
    <span className={`px-3 py-1 rounded-lg text-sm font-medium border ${variants[variant]}`}>
      {children}
    </span>
  );
}

// Card with brand gradient border
export function BrandCard({ children, className = '' }: { children: ReactNode; className?: string }) {
  return (
    <div 
      className={`bg-gray-800/50 backdrop-blur-lg rounded-2xl p-6 ${className}`}
      style={{
        border: '1px solid rgba(140, 0, 255, 0.3)',
      }}
    >
      {children}
    </div>
  );
}
