import { Bell, Search } from 'lucide-react';
import { motion } from 'motion/react';
import { useState } from 'react';

interface DashboardHeaderProps {
  title: string;
  subtitle?: string;
}

export default function DashboardHeader({ title, subtitle }: DashboardHeaderProps) {
  const [showSearch, setShowSearch] = useState(false);

  return (
    <header className="bg-gray-900 border-b border-gray-800 px-4 sm:px-6 lg:px-8 py-4 sticky top-0 z-10 backdrop-blur-xl bg-gray-900/95">
      <div className="flex items-center justify-between gap-4">
        {/* Title Section */}
        <motion.div
          initial={{ opacity: 0, x: -20 }}
          animate={{ opacity: 1, x: 0 }}
          className="flex-1 min-w-0"
        >
          <h1 className="text-white text-xl sm:text-2xl lg:text-3xl font-bold truncate">
            {title}
          </h1>
          {subtitle && (
            <p className="text-gray-400 text-xs sm:text-sm mt-1 truncate">
              {subtitle}
            </p>
          )}
        </motion.div>

        {/* Actions */}
        <div className="flex items-center gap-2 sm:gap-4">
          {/* Search - Hidden on small mobile, icon on medium, full on large */}
          <div className="hidden sm:block">
            {showSearch || window.innerWidth >= 1024 ? (
              <motion.div
                initial={{ width: 0, opacity: 0 }}
                animate={{ width: 'auto', opacity: 1 }}
                className="relative"
              >
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 w-4 h-4 text-gray-400" />
                <input
                  type="text"
                  placeholder="Search..."
                  className="bg-gray-800 text-white pl-10 pr-4 py-2 rounded-lg border border-gray-700 focus:border-purple-500 focus:outline-none w-32 lg:w-64 text-sm"
                  onBlur={() => window.innerWidth < 1024 && setShowSearch(false)}
                  autoFocus={showSearch}
                />
              </motion.div>
            ) : (
              <motion.button
                whileHover={{ scale: 1.05 }}
                whileTap={{ scale: 0.95 }}
                onClick={() => setShowSearch(true)}
                className="p-2 rounded-lg bg-gray-800 hover:bg-gray-700 transition-colors lg:hidden"
              >
                <Search className="w-5 h-5 text-gray-400" />
              </motion.button>
            )}
          </div>

          {/* Notifications */}
          <motion.button
            whileHover={{ scale: 1.05 }}
            whileTap={{ scale: 0.95 }}
            className="relative p-2 rounded-lg bg-gray-800 hover:bg-gray-700 transition-colors"
          >
            <Bell className="w-5 h-5 text-gray-400" />
            <motion.span
              animate={{ scale: [1, 1.2, 1] }}
              transition={{ repeat: Infinity, duration: 2 }}
              className="absolute top-1 right-1 w-2 h-2 bg-red-500 rounded-full"
            />
          </motion.button>
        </div>
      </div>
    </header>
  );
}
