import { Link } from 'react-router';
import { Home, LayoutDashboard, FileText, Users, DollarSign, TrendingUp, Settings, LogOut, Database, Menu, X } from 'lucide-react';
import { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'motion/react';
import ThemeToggle from './ThemeToggle';
import { useTheme } from '../contexts/ThemeContext';

interface SidebarProps {
  role: 'admin' | 'manager';
}

export default function Sidebar({ role }: SidebarProps) {
  const [active, setActive] = useState('dashboard');
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const { primaryColor } = useTheme();

  const adminMenuItems = [
    { id: 'dashboard', label: 'Dashboard', icon: LayoutDashboard, path: '/admin/dashboard' },
    { id: 'manage-data', label: 'Manage Data', icon: Database, path: '/admin/manage-data' },
    { id: 'nda-requests', label: 'NDA Requests', icon: FileText, path: '/admin/dashboard' },
    { id: 'managers', label: 'All Managers', icon: Users, path: '/admin/dashboard' },
    { id: 'payouts', label: 'Payroll', icon: DollarSign, path: '/admin/dashboard' },
    { id: 'reports', label: 'KPI Reports', icon: TrendingUp, path: '/admin/kpi-reports' },
    { id: 'settings', label: 'Settings', icon: Settings, path: '/settings' },
  ];

  const managerMenuItems = [
    { id: 'dashboard', label: 'Dashboard', icon: LayoutDashboard, path: '/manager/dashboard' },
    { id: 'nda', label: 'My NDA', icon: FileText, path: '/manager/dashboard' },
    { id: 'kpi', label: 'KPI Report', icon: TrendingUp, path: '/kpi-report' },
    { id: 'settings', label: 'Settings', icon: Settings, path: '/settings' },
  ];

  const menuItems = role === 'admin' ? adminMenuItems : managerMenuItems;

  // Close mobile menu on resize
  useEffect(() => {
    const handleResize = () => {
      if (window.innerWidth >= 1024) {
        setIsMobileMenuOpen(false);
      }
    };
    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);

  // Prevent scroll when mobile menu is open
  useEffect(() => {
    if (isMobileMenuOpen) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = 'unset';
    }
    return () => {
      document.body.style.overflow = 'unset';
    };
  }, [isMobileMenuOpen]);

  return (
    <>
      {/* Mobile Menu Button - Fixed Top */}
      <motion.button
        initial={{ opacity: 0, y: -20 }}
        animate={{ opacity: 1, y: 0 }}
        onClick={() => setIsMobileMenuOpen(!isMobileMenuOpen)}
        className="lg:hidden fixed top-4 left-4 z-[60] p-3 rounded-xl bg-gray-900 border border-gray-700 shadow-2xl"
        style={{
          background: `linear-gradient(135deg, ${primaryColor}40, #1f293780)`,
          backdropFilter: 'blur(20px)',
        }}
        whileHover={{ scale: 1.05 }}
        whileTap={{ scale: 0.95 }}
      >
        <motion.div
          animate={{ rotate: isMobileMenuOpen ? 90 : 0 }}
          transition={{ duration: 0.3 }}
        >
          {isMobileMenuOpen ? (
            <X className="w-6 h-6 text-white" />
          ) : (
            <Menu className="w-6 h-6 text-white" />
          )}
        </motion.div>
      </motion.button>

      {/* Mobile Menu Overlay */}
      <AnimatePresence>
        {isMobileMenuOpen && (
          <motion.div
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            exit={{ opacity: 0 }}
            transition={{ duration: 0.3 }}
            className="lg:hidden fixed inset-0 bg-black/80 backdrop-blur-sm z-[50]"
            onClick={() => setIsMobileMenuOpen(false)}
          />
        )}
      </AnimatePresence>

      {/* Sidebar - Desktop & Mobile */}
      <motion.div
        initial={{ x: -280 }}
        animate={{ x: isMobileMenuOpen ? 0 : window.innerWidth >= 1024 ? 0 : -280 }}
        transition={{ type: 'spring', damping: 25, stiffness: 200 }}
        className="w-64 h-screen bg-gray-900 border-r border-gray-800 fixed left-0 top-0 flex flex-col z-[55] lg:z-10"
        style={{
          boxShadow: isMobileMenuOpen ? '10px 0 50px rgba(0,0,0,0.5)' : 'none',
        }}
      >
        {/* Logo */}
        <div className="p-6 border-b border-gray-800">
          <Link 
            to="/" 
            className="flex items-center gap-3"
            onClick={() => setIsMobileMenuOpen(false)}
          >
            <motion.div 
              className="w-10 h-10 rounded-lg flex items-center justify-center"
              style={{ 
                background: `linear-gradient(135deg, ${primaryColor}, ${primaryColor}dd)` 
              }}
              whileHover={{ 
                scale: 1.1, 
                rotate: [0, -10, 10, -10, 0],
                transition: { duration: 0.5 }
              }}
            >
              <span className="text-white font-bold text-xl">7★</span>
            </motion.div>
            <div>
              <h1 className="text-white font-bold text-lg">7starswin</h1>
              <p className="text-gray-400 text-xs">{role === 'admin' ? 'Admin Panel' : 'Manager Portal'}</p>
            </div>
          </Link>
        </div>

        {/* Navigation */}
        <nav className="flex-1 p-4 space-y-2 overflow-y-auto">
          {menuItems.map((item, index) => {
            const Icon = item.icon;
            const isActive = active === item.id;
            return (
              <motion.div
                key={item.id}
                initial={{ opacity: 0, x: -20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.05 }}
              >
                <Link
                  to={item.path}
                  onClick={() => {
                    setActive(item.id);
                    setIsMobileMenuOpen(false);
                  }}
                  className={`flex items-center gap-3 px-4 py-3 rounded-lg transition-all group relative overflow-hidden ${
                    isActive
                      ? 'text-white'
                      : 'text-gray-400 hover:bg-gray-800 hover:text-white'
                  }`}
                  style={
                    isActive
                      ? {
                          backgroundColor: primaryColor,
                          boxShadow: `0 10px 25px -5px ${primaryColor}80`,
                        }
                      : {}
                  }
                >
                  {/* 3D Hover Effect */}
                  {!isActive && (
                    <motion.div
                      className="absolute inset-0 bg-gradient-to-r from-transparent via-white/5 to-transparent"
                      initial={{ x: '-100%' }}
                      whileHover={{ x: '100%' }}
                      transition={{ duration: 0.5 }}
                    />
                  )}
                  
                  <motion.div
                    whileHover={{ rotate: [0, -10, 10, 0], scale: 1.1 }}
                    transition={{ duration: 0.3 }}
                  >
                    <Icon className="w-5 h-5" />
                  </motion.div>
                  <span className="font-medium">{item.label}</span>
                  
                  {/* Active Indicator */}
                  {isActive && (
                    <motion.div
                      layoutId="activeTab"
                      className="absolute right-2 w-2 h-2 rounded-full bg-white"
                      initial={{ scale: 0 }}
                      animate={{ scale: 1 }}
                      transition={{ type: 'spring' }}
                    />
                  )}
                </Link>
              </motion.div>
            );
          })}
        </nav>

        {/* Theme Toggle */}
        <div className="px-4 pb-2 border-t border-gray-800 pt-2">
          <ThemeToggle />
        </div>

        {/* User Section */}
        <div className="p-4 border-t border-gray-800">
          <div className="flex items-center gap-3 mb-3">
            <motion.div 
              className="w-10 h-10 rounded-full flex items-center justify-center relative"
              style={{ 
                background: `linear-gradient(135deg, ${primaryColor}, #3b82f6)` 
              }}
              whileHover={{ 
                scale: 1.1,
                boxShadow: `0 0 20px ${primaryColor}80`
              }}
            >
              <span className="text-white font-semibold text-sm">
                {role === 'admin' ? 'AD' : 'MG'}
              </span>
              {/* Online Indicator */}
              <motion.div
                className="absolute -top-1 -right-1 w-3 h-3 bg-emerald-500 rounded-full border-2 border-gray-900"
                animate={{ scale: [1, 1.2, 1] }}
                transition={{ repeat: Infinity, duration: 2 }}
              />
            </motion.div>
            <div className="flex-1">
              <p className="text-white text-sm font-medium">
                {role === 'admin' ? 'Admin User' : 'Manager User'}
              </p>
              <p className="text-gray-400 text-xs capitalize">{role}</p>
            </div>
          </div>
          <motion.button 
            className="flex items-center gap-2 text-gray-400 hover:text-white transition-colors w-full px-4 py-2 rounded-lg hover:bg-gray-800 group"
            whileHover={{ scale: 1.02 }}
            whileTap={{ scale: 0.98 }}
          >
            <motion.div
              whileHover={{ x: -3 }}
              transition={{ type: 'spring', stiffness: 300 }}
            >
              <LogOut className="w-4 h-4" />
            </motion.div>
            <span className="text-sm">Logout</span>
          </motion.button>
        </div>
      </motion.div>
    </>
  );
}
