import { createContext, useContext, useState, useEffect, ReactNode } from 'react';

type ThemeMode = 'site' | 'hr';

interface ThemeContextType {
  mode: ThemeMode;
  toggleTheme: () => void;
  primaryColor: string;
  primaryColorRGB: string;
  primaryDark: string;
}

const ThemeContext = createContext<ThemeContextType | undefined>(undefined);

export function ThemeProvider({ children }: { children: ReactNode }) {
  const [mode, setMode] = useState<ThemeMode>(() => {
    // Load from localStorage or default to 'site'
    const saved = localStorage.getItem('theme-mode');
    return (saved as ThemeMode) || 'site';
  });

  const primaryColor = mode === 'site' ? '#8a2be2' : '#ff00d8';
  const primaryColorRGB = mode === 'site' ? '138, 43, 226' : '255, 0, 216';
  const primaryDark = mode === 'site' ? '#7a1fd2' : '#e600c0';

  useEffect(() => {
    // Save to localStorage
    localStorage.setItem('theme-mode', mode);

    // Update CSS variables
    document.documentElement.style.setProperty('--color-primary', primaryColor);
    document.documentElement.style.setProperty('--color-primary-rgb', primaryColorRGB);
    document.documentElement.style.setProperty('--color-primary-dark', primaryDark);

    // Update body class for global styling
    document.body.classList.remove('theme-site', 'theme-hr');
    document.body.classList.add(`theme-${mode}`);
  }, [mode, primaryColor, primaryColorRGB, primaryDark]);

  const toggleTheme = () => {
    setMode(prev => prev === 'site' ? 'hr' : 'site');
  };

  return (
    <ThemeContext.Provider value={{ mode, toggleTheme, primaryColor, primaryColorRGB, primaryDark }}>
      {children}
    </ThemeContext.Provider>
  );
}

export function useTheme() {
  const context = useContext(ThemeContext);
  if (!context) {
    throw new Error('useTheme must be used within ThemeProvider');
  }
  return context;
}