import { TrendingUp, Users, DollarSign, FileCheck, AlertTriangle, Euro, Upload, Clock, FileText, Eye, CheckCircle, XCircle } from 'lucide-react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from 'recharts';
import { Link } from 'react-router';
import { motion } from 'motion/react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';
import { useEffect, useState } from 'react';

export default function AdminDashboard() {
  const [totals, setTotals] = useState({
    ftd: 0,
    deposits: 0,
    partners: 0,
    managers: 0,
  });

  const kpiData = [
    // Real KPI data will populate here from actual manager performance
  ];

  const managers = [
    // Managers will be added through the admin panel
    // No test data - real managers only
  ];

  const pendingNDAs = [
    // Pending NDA requests will appear here
    // No test data - real applications only
  ];

  // Load totals from localStorage
  useEffect(() => {
    const savedData = localStorage.getItem('7starswin_manager_data');
    if (savedData) {
      const managerList = JSON.parse(savedData);
      const calculated = {
        ftd: managerList.reduce((sum: number, m: any) => sum + m.ftd, 0),
        deposits: managerList.reduce((sum: number, m: any) => sum + m.deposits, 0),
        partners: managerList.reduce((sum: number, m: any) => sum + m.activePartners, 0),
        managers: managerList.length,
      };
      setTotals(calculated);
    }
  }, []);

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 lg:ml-64 w-full">
        <DashboardHeader title="Admin Dashboard" subtitle="Manage your affiliate team and operations" />
        
        <div className="p-4 sm:p-6 lg:p-8 space-y-6">
          {/* Quick Actions Bar */}
          <motion.div
            initial={{ opacity: 0, y: -20 }}
            animate={{ opacity: 1, y: 0 }}
            className="bg-gradient-to-r from-purple-600/20 to-blue-600/20 backdrop-blur-lg border border-purple-500/30 rounded-2xl p-4 sm:p-6"
          >
            <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
              <div>
                <h3 className="text-white font-bold text-base sm:text-lg mb-1">📊 Affiliate Data Integration</h3>
                <p className="text-purple-200 text-xs sm:text-sm">Import KPI data from 7starswin affiliate platform</p>
              </div>
              <Link
                to="/admin/import-kpi-data"
                className="w-full sm:w-auto px-4 sm:px-6 py-3 rounded-xl bg-purple-600 hover:bg-purple-700 text-white font-semibold transition-all flex items-center justify-center gap-2 shadow-lg shadow-purple-600/50"
              >
                <Upload className="w-5 h-5" />
                <span className="text-sm sm:text-base">Sync KPI Data</span>
              </Link>
            </div>
          </motion.div>

          {/* Summary Cards */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              whileHover={{ 
                scale: 1.05, 
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(139, 92, 246, 0.5)"
              }}
              transition={{ type: "spring", stiffness: 300 }}
              className="bg-gradient-to-br from-purple-600/20 to-purple-800/20 backdrop-blur-lg border border-purple-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <motion.div 
                  className="w-12 h-12 rounded-lg bg-purple-600/30 flex items-center justify-center"
                  whileHover={{ rotate: [0, -10, 10, -10, 0] }}
                  transition={{ duration: 0.5 }}
                >
                  <Users className="w-6 h-6 text-purple-400" />
                </motion.div>
                <span className="text-emerald-400 text-sm font-medium">+5</span>
              </div>
              <p className="text-gray-400 text-sm mb-1">Total Managers</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">{totals.managers}</p>
              <p className="text-gray-500 text-xs mt-2">Active managers</p>
            </motion.div>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.1 }}
              whileHover={{ 
                scale: 1.05, 
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(234, 179, 8, 0.5)"
              }}
              className="bg-gradient-to-br from-yellow-600/20 to-yellow-800/20 backdrop-blur-lg border border-yellow-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <motion.div 
                  className="w-12 h-12 rounded-lg bg-yellow-600/30 flex items-center justify-center"
                  whileHover={{ rotate: [0, -10, 10, -10, 0] }}
                  transition={{ duration: 0.5 }}
                >
                  <Clock className="w-6 h-6 text-yellow-400" />
                </motion.div>
                <span className="text-red-400 text-sm font-medium">3 new</span>
              </div>
              <p className="text-gray-400 text-sm mb-1">Pending NDAs</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">0</p>
              <p className="text-gray-500 text-xs mt-2">Awaiting review</p>
            </motion.div>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.2 }}
              whileHover={{ 
                scale: 1.05, 
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(59, 130, 246, 0.5)"
              }}
              className="bg-gradient-to-br from-blue-600/20 to-blue-800/20 backdrop-blur-lg border border-blue-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <motion.div 
                  className="w-12 h-12 rounded-lg bg-blue-600/30 flex items-center justify-center"
                  whileHover={{ rotate: [0, -10, 10, -10, 0] }}
                  transition={{ duration: 0.5 }}
                >
                  <TrendingUp className="w-6 h-6 text-blue-400" />
                </motion.div>
                <span className="text-emerald-400 text-sm font-medium">+18.5%</span>
              </div>
              <p className="text-gray-400 text-sm mb-1">Total FTD</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">{totals.ftd.toLocaleString()}</p>
              <p className="text-gray-500 text-xs mt-2">This month</p>
            </motion.div>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.3 }}
              whileHover={{ 
                scale: 1.05, 
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(16, 185, 129, 0.5)"
              }}
              className="bg-gradient-to-br from-emerald-600/20 to-emerald-800/20 backdrop-blur-lg border border-emerald-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <motion.div 
                  className="w-12 h-12 rounded-lg bg-emerald-600/30 flex items-center justify-center"
                  whileHover={{ rotate: [0, -10, 10, -10, 0] }}
                  transition={{ duration: 0.5 }}
                >
                  <DollarSign className="w-6 h-6 text-emerald-400" />
                </motion.div>
                <span className="text-emerald-400 text-sm font-medium">+22.3%</span>
              </div>
              <p className="text-gray-400 text-sm mb-1">Total Deposits</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">€{totals.deposits.toLocaleString()}</p>
              <p className="text-gray-500 text-xs mt-2">Total deposits</p>
            </motion.div>
          </div>

          {/* Pending NDA Requests */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center justify-between mb-6">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-lg bg-yellow-600/20 flex items-center justify-center">
                  <FileText className="w-5 h-5 text-yellow-400" />
                </div>
                <h3 className="text-white text-xl font-bold">Pending NDA Requests</h3>
              </div>
              <span className="px-3 py-1 rounded-lg bg-yellow-500/20 text-yellow-400 text-sm font-medium">
                {pendingNDAs.length} pending
              </span>
            </div>
            <div className="space-y-3">
              {pendingNDAs.length === 0 ? (
                <div className="text-center py-8">
                    <FileText className="w-12 h-12 mx-auto mb-3 text-gray-600" />
                    <p className="text-gray-400">No pending NDA requests</p>
                    <p className="text-gray-500 text-sm mt-1">New applications will appear here</p>
                </div>
              ) : (
                pendingNDAs.map((nda) => (
                  <div
                    key={nda.id}
                    className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl hover:bg-gray-900/70 transition-colors"
                  >
                    <div className="flex items-center gap-4">
                      <div className="w-10 h-10 rounded-full bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center">
                        <span className="text-white font-semibold text-sm">
                          {nda.name.split(' ').map(n => n[0]).join('')}
                        </span>
                      </div>
                      <div>
                        <p className="text-white font-medium">{nda.name}</p>
                        <p className="text-gray-400 text-sm">{nda.country}</p>
                      </div>
                    </div>
                    <div className="flex items-center gap-4">
                      <p className="text-gray-400 text-sm">Applied: {nda.appliedDate}</p>
                      <Link
                        to={`/admin/nda-review/${nda.id}`}
                        className="px-4 py-2 rounded-lg bg-purple-600 hover:bg-purple-700 text-white text-sm font-medium transition-colors flex items-center gap-2"
                      >
                        <Eye className="w-4 h-4" />
                        Review
                      </Link>
                    </div>
                  </div>
                ))
              )}
            </div>
          </div>

          {/* KPI Overview Chart */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center justify-between mb-6">
              <h3 className="text-white text-xl font-bold">Monthly KPI Overview</h3>
              <select className="bg-gray-700/50 border border-gray-600 rounded-lg px-4 py-2 text-white text-sm focus:outline-none focus:border-purple-500">
                <option>Last 6 months</option>
                <option>Last 12 months</option>
                <option>This year</option>
              </select>
            </div>
            <ResponsiveContainer width="100%" height={300}>
              <BarChart data={kpiData}>
                <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                <XAxis dataKey="month" stroke="#9CA3AF" />
                <YAxis stroke="#9CA3AF" />
                <Tooltip
                  contentStyle={{
                    backgroundColor: '#1F2937',
                    border: '1px solid #374151',
                    borderRadius: '0.5rem',
                    color: '#fff',
                  }}
                />
                <Bar dataKey="ftd" fill="#8B5CF6" radius={[8, 8, 0, 0]} />
                <Bar dataKey="deposits" fill="#10B981" radius={[8, 8, 0, 0]} />
              </BarChart>
            </ResponsiveContainer>
            <div className="flex items-center justify-center gap-6 mt-4">
              <div className="flex items-center gap-2">
                <div className="w-3 h-3 rounded-full bg-purple-600"></div>
                <span className="text-gray-400 text-sm">FTD</span>
              </div>
              <div className="flex items-center gap-2">
                <div className="w-3 h-3 rounded-full bg-emerald-600"></div>
                <span className="text-gray-400 text-sm">Deposits ($1K)</span>
              </div>
            </div>
          </div>

          {/* All Managers Table */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center justify-between mb-6">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-lg bg-purple-600/20 flex items-center justify-center">
                  <Users className="w-5 h-5 text-purple-400" />
                </div>
                <h3 className="text-white text-xl font-bold">All Managers</h3>
              </div>
              <Link
                to="/admin/create-user"
                className="px-4 py-2 rounded-lg bg-purple-600 hover:bg-purple-700 text-white text-sm font-medium transition-colors"
              >
                + Add Manager
              </Link>
            </div>
            <div className="overflow-x-auto">
              {managers.length === 0 ? (
                <div className="text-center py-12">
                  <Users className="w-12 h-12 mx-auto mb-3 text-gray-600" />
                  <p className="text-gray-400 mb-2">No managers yet</p>
                  <p className="text-gray-500 text-sm mb-4">Add your first manager to get started</p>
                  <Link
                    to="/admin/create-user"
                    className="inline-block px-4 py-2 rounded-lg bg-purple-600 hover:bg-purple-700 text-white text-sm font-medium transition-colors"
                  >
                    + Add Manager
                  </Link>
                </div>
              ) : (
                <table className="w-full">
                  <thead>
                    <tr className="border-b border-gray-700">
                      <th className="text-left text-gray-400 text-sm font-medium pb-4 px-4">Name</th>
                      <th className="text-left text-gray-400 text-sm font-medium pb-4 px-4">Country</th>
                      <th className="text-left text-gray-400 text-sm font-medium pb-4 px-4">FTD</th>
                      <th className="text-left text-gray-400 text-sm font-medium pb-4 px-4">Deposits</th>
                      <th className="text-left text-gray-400 text-sm font-medium pb-4 px-4">Salary</th>
                      <th className="text-left text-gray-400 text-sm font-medium pb-4 px-4">NDA Status</th>
                      <th className="text-left text-gray-400 text-sm font-medium pb-4 px-4">Actions</th>
                    </tr>
                  </thead>
                  <tbody>
                    {managers.map((manager) => (
                      <tr key={manager.id} className="border-b border-gray-700/50 hover:bg-gray-900/30">
                        <td className="py-4 px-4">
                          <div className="flex items-center gap-3">
                            <div className="w-8 h-8 rounded-full bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center">
                              <span className="text-white text-xs font-semibold">
                                {manager.name.split(' ').map(n => n[0]).join('')}
                              </span>
                            </div>
                            <span className="text-white font-medium">{manager.name}</span>
                          </div>
                        </td>
                        <td className="py-4 px-4">
                          <span className="text-gray-300">{manager.country}</span>
                        </td>
                        <td className="py-4 px-4">
                          <span className="text-white font-semibold">{manager.ftd}</span>
                        </td>
                        <td className="py-4 px-4">
                          <span className="text-white font-semibold">${manager.totalDeposit.toLocaleString()}</span>
                        </td>
                        <td className="py-4 px-4">
                          <span className="text-emerald-400 font-semibold">${manager.salary.toLocaleString()}</span>
                        </td>
                        <td className="py-4 px-4">
                          {manager.status === 'approved' && (
                            <span className="px-3 py-1 rounded-lg bg-emerald-500/20 text-emerald-400 text-xs font-medium inline-flex items-center gap-1">
                              <CheckCircle className="w-3 h-3" />
                              Approved
                            </span>
                          )}
                          {manager.status === 'pending' && (
                            <span className="px-3 py-1 rounded-lg bg-yellow-500/20 text-yellow-400 text-xs font-medium inline-flex items-center gap-1">
                              <Clock className="w-3 h-3" />
                              Pending
                            </span>
                          )}
                          {manager.status === 'rejected' && (
                            <span className="px-3 py-1 rounded-lg bg-red-500/20 text-red-400 text-xs font-medium inline-flex items-center gap-1">
                              <XCircle className="w-3 h-3" />
                              Rejected
                            </span>
                          )}
                        </td>
                        <td className="py-4 px-4">
                          <div className="flex items-center gap-2">
                            <Link
                              to={`/admin/manager/${manager.id}`}
                              className="px-3 py-1.5 rounded-lg bg-purple-600 hover:bg-purple-700 text-white text-xs font-medium transition-colors"
                            >
                              View
                            </Link>
                            <Link
                              to={`/admin/payout/${manager.id}`}
                              className="px-3 py-1.5 rounded-lg bg-emerald-600 hover:bg-emerald-700 text-white text-xs font-medium transition-colors"
                            >
                              Payout
                            </Link>
                          </div>
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              )}
            </div>
          </div>

          {/* Payroll Summary */}
          <div className="grid md:grid-cols-3 gap-6">
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <div className="flex items-center gap-3 mb-4">
                <div className="w-10 h-10 rounded-lg bg-emerald-600/20 flex items-center justify-center">
                  <DollarSign className="w-5 h-5 text-emerald-400" />
                </div>
                <h3 className="text-white text-lg font-bold">This Month</h3>
              </div>
              <p className="text-white text-3xl font-bold mb-2">$135,000</p>
              <p className="text-gray-400 text-sm">Total payroll</p>
            </div>

            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <div className="flex items-center gap-3 mb-4">
                <div className="w-10 h-10 rounded-lg bg-blue-600/20 flex items-center justify-center">
                  <DollarSign className="w-5 h-5 text-blue-400" />
                </div>
                <h3 className="text-white text-lg font-bold">Last Month</h3>
              </div>
              <p className="text-white text-3xl font-bold mb-2">$128,500</p>
              <p className="text-gray-400 text-sm">Total paid out</p>
            </div>

            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <div className="flex items-center gap-3 mb-4">
                <div className="w-10 h-10 rounded-lg bg-purple-600/20 flex items-center justify-center">
                  <TrendingUp className="w-5 h-5 text-purple-400" />
                </div>
                <h3 className="text-white text-lg font-bold">Growth</h3>
              </div>
              <p className="text-white text-3xl font-bold mb-2">+5.1%</p>
              <p className="text-gray-400 text-sm">Month over month</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}