import { useState } from 'react';
import { Download, Calendar, TrendingUp, Target, Euro, Users, UserCircle, ChevronDown } from 'lucide-react';
import { BarChart, Bar, LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { motion } from 'motion/react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

export default function AdminKPIReports() {
  const [selectedMonth, setSelectedMonth] = useState('2026-06');
  const [selectedManager, setSelectedManager] = useState('all');

  const managers = [
    { id: 'all', name: 'All Managers (Combined)', email: 'all@managers' },
    // Real managers will appear here after admin creates them
  ];

  // Data changes based on selected manager
  const getManagerData = () => {
    if (selectedManager === 'all') {
      // Combined data for all managers - will populate with real data
      return {
        monthlyData: [],
        performanceData: [],
        weeklyData: [],
        summary: {
          totalFTD: 0,
          totalPartners: 0,
          totalDeposits: 0,
          avgMonthly: 0,
        },
      };
    } else {
      // Individual manager data - will populate with real data
      return {
        monthlyData: [],
        performanceData: [],
        weeklyData: [],
        summary: {
          totalFTD: 0,
          totalPartners: 0,
          totalDeposits: 0,
          avgMonthly: 0,
        },
      };
    }
  };

  const data = getManagerData();
  const currentManager = managers.find(m => m.id === selectedManager);

  const handleExportPDF = () => {
    alert(`Exporting KPI report for ${currentManager?.name}...`);
  };

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 ml-64">
        <DashboardHeader title="KPI Reports" subtitle="Comprehensive performance analytics for all managers" />
        
        <div className="p-8 space-y-6">
          {/* Manager Selection & Actions */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-4">
                <div className="w-12 h-12 rounded-lg bg-purple-600/20 flex items-center justify-center">
                  <TrendingUp className="w-6 h-6 text-purple-400" />
                </div>
                <div>
                  <h2 className="text-white text-xl font-bold">Performance Report</h2>
                  <p className="text-gray-400 text-sm">Select manager to view detailed KPI analysis</p>
                </div>
              </div>
              <div className="flex items-center gap-4">
                {/* Manager Selector */}
                <div className="relative">
                  <select
                    value={selectedManager}
                    onChange={(e) => setSelectedManager(e.target.value)}
                    className="appearance-none bg-gradient-to-r from-purple-600 to-purple-700 border-2 border-purple-500/50 rounded-xl pl-4 pr-12 py-3 text-white font-semibold focus:outline-none focus:border-purple-400 cursor-pointer min-w-[280px]"
                  >
                    {managers.map((manager) => (
                      <option key={manager.id} value={manager.id} className="bg-gray-800">
                        {manager.name}
                      </option>
                    ))}
                  </select>
                  <ChevronDown className="absolute right-4 top-1/2 transform -translate-y-1/2 w-5 h-5 text-white pointer-events-none" />
                </div>

                {/* Month Selector */}
                <div className="flex items-center gap-2">
                  <Calendar className="w-5 h-5 text-gray-400" />
                  <select
                    value={selectedMonth}
                    onChange={(e) => setSelectedMonth(e.target.value)}
                    className="bg-gray-700/50 border border-gray-600 rounded-lg px-4 py-2 text-white focus:outline-none focus:border-purple-500"
                  >
                    <option value="2026-06">June 2026</option>
                    <option value="2026-05">May 2026</option>
                    <option value="2026-04">April 2026</option>
                    <option value="2026-03">March 2026</option>
                    <option value="2026-02">February 2026</option>
                    <option value="2026-01">January 2026</option>
                  </select>
                </div>

                {/* Export Button */}
                <button
                  onClick={handleExportPDF}
                  className="flex items-center gap-2 px-6 py-2 rounded-lg bg-emerald-600 hover:bg-emerald-700 text-white font-semibold shadow-lg shadow-emerald-600/50 transition-all"
                >
                  <Download className="w-5 h-5" />
                  Export PDF
                </button>
              </div>
            </div>
          </div>

          {/* Current Selection Info */}
          {selectedManager !== 'all' && currentManager && (
            <div className="bg-gradient-to-r from-blue-600/20 to-purple-600/20 backdrop-blur-lg border border-blue-500/30 rounded-2xl p-4">
              <div className="flex items-center gap-4">
                <div className="w-12 h-12 rounded-full bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center">
                  <span className="text-white font-bold">
                    {currentManager.name.split(' ').map(n => n[0]).join('')}
                  </span>
                </div>
                <div>
                  <p className="text-white font-semibold text-lg">{currentManager.name}</p>
                  <p className="text-blue-200 text-sm">{currentManager.email}</p>
                </div>
                <div className="ml-auto">
                  <span className="px-4 py-2 rounded-lg bg-emerald-500/20 text-emerald-300 text-sm font-medium">
                    ✓ NDA Approved
                  </span>
                </div>
              </div>
            </div>
          )}

          {/* Performance Summary Cards */}
          <div className="grid md:grid-cols-3 gap-6">
            {data.performanceData.map((item, index) => (
              <div
                key={index}
                className="bg-pink-600/10 backdrop-blur-lg border border-pink-500/30 rounded-2xl p-6"
              >
                <div className="flex items-center justify-between mb-4">
                  <h3 className="text-white font-semibold">{item.metric}</h3>
                  <div style={{ perspective: '1000px' }}>
                    <motion.div
                      className="w-10 h-10 rounded-lg bg-purple-600/20 flex items-center justify-center cursor-pointer"
                      style={{ transformStyle: 'preserve-3d' }}
                      animate={{
                        rotateY: [0, 360],
                        scale: [1, 1.1, 1],
                      }}
                      transition={{
                        duration: 3,
                        repeat: Infinity,
                        ease: "easeInOut",
                      }}
                      whileHover={{
                        scale: 1.2,
                        rotateX: 15,
                        rotateY: 15,
                      }}
                    >
                      <motion.div
                        animate={{
                          z: [0, 20, 0],
                        }}
                        transition={{
                          duration: 3,
                          repeat: Infinity,
                          ease: "easeInOut",
                        }}
                      >
                        {item.metric === 'FTD' && (
                          <Users className="w-5 h-5 text-purple-400" style={{ filter: 'drop-shadow(0 4px 8px rgba(139, 92, 246, 0.6))' }} />
                        )}
                        {item.metric === 'Partners' && (
                          <UserCircle className="w-5 h-5 text-purple-400" style={{ filter: 'drop-shadow(0 4px 8px rgba(139, 92, 246, 0.6))' }} />
                        )}
                        {item.metric === 'Deposits' && (
                          <Euro className="w-5 h-5 text-purple-400" style={{ filter: 'drop-shadow(0 4px 8px rgba(139, 92, 246, 0.6))' }} />
                        )}
                      </motion.div>
                    </motion.div>
                  </div>
                </div>
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <span className="text-gray-400 text-sm">Achieved</span>
                    <span className="text-white text-xl font-bold">
                      {item.metric === 'Deposits' ? `$${item.achieved.toLocaleString()}` : item.achieved}
                    </span>
                  </div>
                  <div className="flex items-center justify-between">
                    <span className="text-gray-400 text-sm">Target</span>
                    <span className="text-gray-300 text-lg">
                      {item.metric === 'Deposits' ? `$${item.target.toLocaleString()}` : item.target}
                    </span>
                  </div>
                  <div className="pt-3 border-t border-gray-700">
                    <div className="flex items-center justify-between mb-2">
                      <span className="text-gray-400 text-sm">Performance</span>
                      <span
                        className={`text-lg font-bold ${
                          item.percentage >= 100 ? 'text-emerald-400' : 'text-yellow-400'
                        }`}
                      >
                        {item.percentage}%
                      </span>
                    </div>
                    <div className="w-full h-2 bg-gray-700 rounded-full overflow-hidden">
                      <div
                        className={`h-full rounded-full ${
                          item.percentage >= 100
                            ? 'bg-gradient-to-r from-emerald-600 to-emerald-400'
                            : 'bg-gradient-to-r from-yellow-600 to-yellow-400'
                        }`}
                        style={{ width: `${Math.min(item.percentage, 100)}%` }}
                      />
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>

          {/* FTD: Target vs Achieved */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <h3 className="text-white text-xl font-bold mb-6">FTD: Target vs Achieved</h3>
            <ResponsiveContainer width="100%" height={350}>
              <BarChart data={data.monthlyData}>
                <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                <XAxis dataKey="month" stroke="#9CA3AF" />
                <YAxis stroke="#9CA3AF" />
                <Tooltip
                  contentStyle={{
                    backgroundColor: '#1F2937',
                    border: '1px solid #374151',
                    borderRadius: '0.5rem',
                    color: '#fff',
                  }}
                />
                <Legend />
                <Bar dataKey="target" fill="#6B7280" name="Target" radius={[8, 8, 0, 0]} />
                <Bar dataKey="ftd" fill="#8B5CF6" name="Achieved" radius={[8, 8, 0, 0]} />
              </BarChart>
            </ResponsiveContainer>
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {/* Deposit Trend */}
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <h3 className="text-white text-xl font-bold mb-6">Weekly Deposit Trend</h3>
              <ResponsiveContainer width="100%" height={300}>
                <LineChart data={data.weeklyData}>
                  <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                  <XAxis dataKey="week" stroke="#9CA3AF" />
                  <YAxis stroke="#9CA3AF" />
                  <Tooltip
                    contentStyle={{
                      backgroundColor: '#1F2937',
                      border: '1px solid #374151',
                      borderRadius: '0.5rem',
                      color: '#fff',
                    }}
                  />
                  <Line
                    type="monotone"
                    dataKey="deposits"
                    stroke="#10B981"
                    strokeWidth={3}
                    dot={{ fill: '#10B981', r: 6 }}
                  />
                </LineChart>
              </ResponsiveContainer>
            </div>

            {/* Partner Growth */}
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <h3 className="text-white text-xl font-bold mb-6">Partner Growth</h3>
              <ResponsiveContainer width="100%" height={300}>
                <BarChart data={data.monthlyData}>
                  <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                  <XAxis dataKey="month" stroke="#9CA3AF" />
                  <YAxis stroke="#9CA3AF" />
                  <Tooltip
                    contentStyle={{
                      backgroundColor: '#1F2937',
                      border: '1px solid #374151',
                      borderRadius: '0.5rem',
                      color: '#fff',
                    }}
                  />
                  <Bar dataKey="partners" fill="#3B82F6" radius={[8, 8, 0, 0]} />
                </BarChart>
              </ResponsiveContainer>
            </div>
          </div>

          {/* Summary Statistics */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <h3 className="text-white text-xl font-bold mb-6">6-Month Summary</h3>
            <div className="grid md:grid-cols-4 gap-6">
              <div className="p-6 bg-gradient-to-br from-purple-600/20 to-purple-800/20 border border-purple-500/30 rounded-xl text-center">
                <p className="text-gray-400 text-sm mb-2">Total FTD</p>
                <p className="text-white text-3xl font-bold">{data.summary.totalFTD.toLocaleString()}</p>
                <p className="text-emerald-400 text-sm mt-2">
                  {selectedManager === 'all' ? 'All managers combined' : '+75% vs target'}
                </p>
              </div>
              <div className="p-6 bg-gradient-to-br from-blue-600/20 to-blue-800/20 border border-blue-500/30 rounded-xl text-center">
                <p className="text-gray-400 text-sm mb-2">Total Partners</p>
                <p className="text-white text-3xl font-bold">{data.summary.totalPartners.toLocaleString()}</p>
                <p className="text-emerald-400 text-sm mt-2">
                  {selectedManager === 'all' ? 'Active partners' : '+42% growth'}
                </p>
              </div>
              <div className="p-6 bg-gradient-to-br from-emerald-600/20 to-emerald-800/20 border border-emerald-500/30 rounded-xl text-center">
                <p className="text-gray-400 text-sm mb-2">Total Deposits</p>
                <p className="text-white text-3xl font-bold">${(data.summary.totalDeposits / 1000).toFixed(0)}K</p>
                <p className="text-emerald-400 text-sm mt-2">+86% increase</p>
              </div>
              <div className="p-6 bg-gradient-to-br from-amber-600/20 to-amber-800/20 border border-amber-500/30 rounded-xl text-center">
                <p className="text-gray-400 text-sm mb-2">Avg Monthly</p>
                <p className="text-white text-3xl font-bold">${(data.summary.avgMonthly / 1000).toFixed(1)}K</p>
                <p className="text-gray-400 text-sm mt-2">Per month</p>
              </div>
            </div>
          </div>

          {/* Performance Insights */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <h3 className="text-white text-xl font-bold mb-6">Performance Insights</h3>
            <div className="space-y-4">
              {selectedManager === 'all' ? (
                <>
                  <div className="p-4 bg-emerald-500/10 border border-emerald-500/30 rounded-xl">
                    <div className="flex items-start gap-3">
                      <div className="w-6 h-6 rounded-full bg-emerald-600 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <span className="text-white text-xs">✓</span>
                      </div>
                      <div>
                        <p className="text-emerald-400 font-semibold mb-1">Team Excellence</p>
                        <p className="text-gray-300 text-sm">
                          Your team has exceeded the combined FTD target by 31% this month. Outstanding performance across all managers!
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-blue-500/10 border border-blue-500/30 rounded-xl">
                    <div className="flex items-start gap-3">
                      <div className="w-6 h-6 rounded-full bg-blue-600 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <span className="text-white text-xs">ℹ</span>
                      </div>
                      <div>
                        <p className="text-blue-400 font-semibold mb-1">Consistent Growth</p>
                        <p className="text-gray-300 text-sm">
                          The combined partner network has grown by 18% this quarter, showing strong team-wide expansion.
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-purple-500/10 border border-purple-500/30 rounded-xl">
                    <div className="flex items-start gap-3">
                      <div className="w-6 h-6 rounded-full bg-purple-600 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <span className="text-white text-xs">★</span>
                      </div>
                      <div>
                        <p className="text-purple-400 font-semibold mb-1">Top Performing Team</p>
                        <p className="text-gray-300 text-sm">
                          Your affiliate team is performing in the top tier with excellent deposit volumes and conversion rates.
                        </p>
                      </div>
                    </div>
                  </div>
                </>
              ) : (
                <>
                  <div className="p-4 bg-emerald-500/10 border border-emerald-500/30 rounded-xl">
                    <div className="flex items-start gap-3">
                      <div className="w-6 h-6 rounded-full bg-emerald-600 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <span className="text-white text-xs">✓</span>
                      </div>
                      <div>
                        <p className="text-emerald-400 font-semibold mb-1">Excellent Performance</p>
                        <p className="text-gray-300 text-sm">
                          {currentManager?.name} has exceeded their FTD target by 23% this month. Keep up the great work!
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-blue-500/10 border border-blue-500/30 rounded-xl">
                    <div className="flex items-start gap-3">
                      <div className="w-6 h-6 rounded-full bg-blue-600 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <span className="text-white text-xs">ℹ</span>
                      </div>
                      <div>
                        <p className="text-blue-400 font-semibold mb-1">Consistent Growth</p>
                        <p className="text-gray-300 text-sm">
                          Partner count has grown steadily over the past 6 months, showing strong network building.
                        </p>
                      </div>
                    </div>
                  </div>
                  <div className="p-4 bg-purple-500/10 border border-purple-500/30 rounded-xl">
                    <div className="flex items-start gap-3">
                      <div className="w-6 h-6 rounded-full bg-purple-600 flex items-center justify-center flex-shrink-0 mt-0.5">
                        <span className="text-white text-xs">★</span>
                      </div>
                      <div>
                        <p className="text-purple-400 font-semibold mb-1">Top Performer</p>
                        <p className="text-gray-300 text-sm">
                          This manager is in the top 10% based on deposit volume and conversion rates.
                        </p>
                      </div>
                    </div>
                  </div>
                </>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}