import { useState, useEffect } from 'react';
import { motion } from 'motion/react';
import { Users, TrendingUp, DollarSign, Target, Save, Plus, Trash2, Edit2, Euro } from 'lucide-react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

interface ManagerData {
  id: string;
  name: string;
  email: string;
  ftd: number;
  deposits: number;
  activePartners: number;
  salary: number;
}

export default function AdminManageData() {
  const [managers, setManagers] = useState<ManagerData[]>([]);
  const [editingId, setEditingId] = useState<string | null>(null);
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    ftd: 0,
    deposits: 0,
    activePartners: 0,
    salary: 0,
  });
  const [showAddForm, setShowAddForm] = useState(false);

  // Load data from localStorage
  useEffect(() => {
    const savedData = localStorage.getItem('7starswin_manager_data');
    if (savedData) {
      setManagers(JSON.parse(savedData));
    }
  }, []);

  // Save data to localStorage
  const saveData = (data: ManagerData[]) => {
    localStorage.setItem('7starswin_manager_data', JSON.stringify(data));
    setManagers(data);
  };

  const handleAdd = () => {
    if (!formData.name || !formData.email) {
      alert('Please fill in name and email');
      return;
    }

    const newManager: ManagerData = {
      id: Date.now().toString(),
      ...formData,
    };

    const updatedManagers = [...managers, newManager];
    saveData(updatedManagers);
    resetForm();
    setShowAddForm(false);
  };

  const handleEdit = (manager: ManagerData) => {
    setEditingId(manager.id);
    setFormData({
      name: manager.name,
      email: manager.email,
      ftd: manager.ftd,
      deposits: manager.deposits,
      activePartners: manager.activePartners,
      salary: manager.salary,
    });
  };

  const handleUpdate = () => {
    if (!editingId) return;

    const updatedManagers = managers.map((m) =>
      m.id === editingId
        ? { ...m, ...formData }
        : m
    );

    saveData(updatedManagers);
    setEditingId(null);
    resetForm();
  };

  const handleDelete = (id: string) => {
    if (confirm('Are you sure you want to delete this manager data?')) {
      const updatedManagers = managers.filter((m) => m.id !== id);
      saveData(updatedManagers);
    }
  };

  const resetForm = () => {
    setFormData({
      name: '',
      email: '',
      ftd: 0,
      deposits: 0,
      activePartners: 0,
      salary: 0,
    });
    setEditingId(null);
  };

  // Calculate totals
  const totals = {
    ftd: managers.reduce((sum, m) => sum + m.ftd, 0),
    deposits: managers.reduce((sum, m) => sum + m.deposits, 0),
    partners: managers.reduce((sum, m) => sum + m.activePartners, 0),
    salary: managers.reduce((sum, m) => sum + m.salary, 0),
  };

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 lg:ml-64 w-full">
        <DashboardHeader 
          title="Manage Manager Data" 
          subtitle="Add and customize FTD, deposits, partners, and salary data for each manager" 
        />
        
        <div className="p-4 sm:p-6 lg:p-8 space-y-6">
          {/* Summary Cards */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6">
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              whileHover={{ 
                scale: 1.05,
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(139, 92, 246, 0.5)"
              }}
              transition={{ type: "spring", stiffness: 300 }}
              className="bg-gradient-to-br from-purple-600/20 to-purple-800/20 backdrop-blur-lg border border-purple-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <TrendingUp className="w-10 h-10 text-purple-400" />
              </div>
              <p className="text-gray-400 text-sm mb-1">Total FTD (All Managers)</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">{totals.ftd.toLocaleString()}</p>
            </motion.div>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.1 }}
              whileHover={{ 
                scale: 1.05,
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(16, 185, 129, 0.5)"
              }}
              className="bg-gradient-to-br from-emerald-600/20 to-emerald-800/20 backdrop-blur-lg border border-emerald-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <DollarSign className="w-10 h-10 text-emerald-400" />
              </div>
              <p className="text-gray-400 text-sm mb-1">Total Deposits</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">€{totals.deposits.toLocaleString()}</p>
            </motion.div>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.2 }}
              whileHover={{ 
                scale: 1.05,
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(59, 130, 246, 0.5)"
              }}
              className="bg-gradient-to-br from-blue-600/20 to-blue-800/20 backdrop-blur-lg border border-blue-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <Target className="w-10 h-10 text-blue-400" />
              </div>
              <p className="text-gray-400 text-sm mb-1">Active Partners</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">{totals.partners.toLocaleString()}</p>
            </motion.div>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.3 }}
              whileHover={{ 
                scale: 1.05,
                rotateY: 5,
                boxShadow: "0 25px 50px -12px rgba(245, 158, 11, 0.5)"
              }}
              className="bg-gradient-to-br from-amber-600/20 to-amber-800/20 backdrop-blur-lg border border-amber-500/30 rounded-2xl p-6"
              style={{ transformStyle: 'preserve-3d' }}
            >
              <div className="flex items-center justify-between mb-4">
                <Euro className="w-10 h-10 text-amber-400" />
              </div>
              <p className="text-gray-400 text-sm mb-1">Total Salary Budget</p>
              <p className="text-white text-2xl sm:text-3xl font-bold">€{totals.salary.toLocaleString()}</p>
            </motion.div>
          </div>

          {/* Add Manager Button */}
          <div className="flex justify-end">
            <motion.button
              whileHover={{ scale: 1.05 }}
              whileTap={{ scale: 0.95 }}
              onClick={() => {
                setShowAddForm(!showAddForm);
                resetForm();
              }}
              className="flex items-center gap-2 px-4 sm:px-6 py-3 bg-gradient-to-r from-purple-600 to-purple-700 hover:from-purple-700 hover:to-purple-800 text-white font-semibold rounded-xl transition-all shadow-lg"
            >
              <Plus className="w-5 h-5" />
              <span className="hidden sm:inline">Add New Manager</span>
              <span className="sm:hidden">Add Manager</span>
            </motion.button>
          </div>

          {/* Add/Edit Form */}
          {(showAddForm || editingId) && (
            <motion.div
              initial={{ opacity: 0, y: -20 }}
              animate={{ opacity: 1, y: 0 }}
              className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6"
            >
              <h3 className="text-white text-xl font-bold mb-6">
                {editingId ? 'Edit Manager Data' : 'Add New Manager'}
              </h3>
              
              <div className="grid md:grid-cols-2 gap-6">
                <div>
                  <label className="block text-gray-400 text-sm font-medium mb-2">
                    Manager Name *
                  </label>
                  <input
                    type="text"
                    value={formData.name}
                    onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                    className="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 rounded-xl text-white focus:border-purple-500 focus:outline-none"
                    placeholder="John Doe"
                  />
                </div>

                <div>
                  <label className="block text-gray-400 text-sm font-medium mb-2">
                    Email Address *
                  </label>
                  <input
                    type="email"
                    value={formData.email}
                    onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                    className="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 rounded-xl text-white focus:border-purple-500 focus:outline-none"
                    placeholder="john@example.com"
                  />
                </div>

                <div>
                  <label className="block text-gray-400 text-sm font-medium mb-2">
                    FTD (First Time Deposits)
                  </label>
                  <input
                    type="number"
                    value={formData.ftd}
                    onChange={(e) => setFormData({ ...formData, ftd: parseInt(e.target.value) || 0 })}
                    className="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 rounded-xl text-white focus:border-purple-500 focus:outline-none"
                    placeholder="210"
                  />
                </div>

                <div>
                  <label className="block text-gray-400 text-sm font-medium mb-2">
                    Total Deposits (€)
                  </label>
                  <input
                    type="number"
                    value={formData.deposits}
                    onChange={(e) => setFormData({ ...formData, deposits: parseInt(e.target.value) || 0 })}
                    className="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 rounded-xl text-white focus:border-purple-500 focus:outline-none"
                    placeholder="82000"
                  />
                </div>

                <div>
                  <label className="block text-gray-400 text-sm font-medium mb-2">
                    Active Partners
                  </label>
                  <input
                    type="number"
                    value={formData.activePartners}
                    onChange={(e) => setFormData({ ...formData, activePartners: parseInt(e.target.value) || 0 })}
                    className="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 rounded-xl text-white focus:border-purple-500 focus:outline-none"
                    placeholder="35"
                  />
                </div>

                <div>
                  <label className="block text-gray-400 text-sm font-medium mb-2">
                    Monthly Salary (€)
                  </label>
                  <input
                    type="number"
                    value={formData.salary}
                    onChange={(e) => setFormData({ ...formData, salary: parseInt(e.target.value) || 0 })}
                    className="w-full px-4 py-3 bg-gray-700/50 border border-gray-600 rounded-xl text-white focus:border-purple-500 focus:outline-none"
                    placeholder="6000"
                  />
                </div>
              </div>

              <div className="flex gap-4 mt-6">
                <button
                  onClick={editingId ? handleUpdate : handleAdd}
                  className="flex items-center gap-2 px-6 py-3 bg-gradient-to-r from-emerald-600 to-emerald-700 hover:from-emerald-700 hover:to-emerald-800 text-white font-semibold rounded-xl transition-all"
                >
                  <Save className="w-5 h-5" />
                  {editingId ? 'Update' : 'Save'}
                </button>
                <button
                  onClick={() => {
                    resetForm();
                    setShowAddForm(false);
                  }}
                  className="px-6 py-3 bg-gray-700 hover:bg-gray-600 text-white font-semibold rounded-xl transition-all"
                >
                  Cancel
                </button>
              </div>
            </motion.div>
          )}

          {/* Managers List */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl overflow-hidden">
            <div className="p-6 border-b border-gray-700">
              <h3 className="text-white text-xl font-bold flex items-center gap-2">
                <Users className="w-6 h-6 text-purple-400" />
                All Managers ({managers.length})
              </h3>
            </div>

            {managers.length === 0 ? (
              <div className="p-12 text-center">
                <Users className="w-16 h-16 text-gray-600 mx-auto mb-4" />
                <p className="text-gray-400 text-lg mb-2">No managers added yet</p>
                <p className="text-gray-500 text-sm">Click "Add New Manager" to get started</p>
              </div>
            ) : (
              <div className="overflow-x-auto">
                <table className="w-full">
                  <thead className="bg-gray-700/50">
                    <tr>
                      <th className="px-6 py-4 text-left text-gray-400 text-sm font-semibold">Manager</th>
                      <th className="px-6 py-4 text-left text-gray-400 text-sm font-semibold">FTD</th>
                      <th className="px-6 py-4 text-left text-gray-400 text-sm font-semibold">Deposits</th>
                      <th className="px-6 py-4 text-left text-gray-400 text-sm font-semibold">Partners</th>
                      <th className="px-6 py-4 text-left text-gray-400 text-sm font-semibold">Salary</th>
                      <th className="px-6 py-4 text-right text-gray-400 text-sm font-semibold">Actions</th>
                    </tr>
                  </thead>
                  <tbody>
                    {managers.map((manager, index) => (
                      <motion.tr
                        key={manager.id}
                        initial={{ opacity: 0 }}
                        animate={{ opacity: 1 }}
                        transition={{ delay: index * 0.05 }}
                        className="border-t border-gray-700 hover:bg-gray-700/30 transition-colors"
                      >
                        <td className="px-6 py-4">
                          <div>
                            <p className="text-white font-semibold">{manager.name}</p>
                            <p className="text-gray-400 text-sm">{manager.email}</p>
                          </div>
                        </td>
                        <td className="px-6 py-4">
                          <p className="text-white font-semibold">{manager.ftd.toLocaleString()}</p>
                        </td>
                        <td className="px-6 py-4">
                          <p className="text-emerald-400 font-semibold">€{manager.deposits.toLocaleString()}</p>
                        </td>
                        <td className="px-6 py-4">
                          <p className="text-blue-400 font-semibold">{manager.activePartners}</p>
                        </td>
                        <td className="px-6 py-4">
                          <p className="text-amber-400 font-semibold">€{manager.salary.toLocaleString()}</p>
                        </td>
                        <td className="px-6 py-4">
                          <div className="flex items-center justify-end gap-2">
                            <button
                              onClick={() => handleEdit(manager)}
                              className="p-2 bg-blue-600/20 hover:bg-blue-600/40 text-blue-400 rounded-lg transition-all"
                              title="Edit"
                            >
                              <Edit2 className="w-4 h-4" />
                            </button>
                            <button
                              onClick={() => handleDelete(manager.id)}
                              className="p-2 bg-red-600/20 hover:bg-red-600/40 text-red-400 rounded-lg transition-all"
                              title="Delete"
                            >
                              <Trash2 className="w-4 h-4" />
                            </button>
                          </div>
                        </td>
                      </motion.tr>
                    ))}
                  </tbody>
                </table>
              </div>
            )}
          </div>

          {/* Instructions */}
          <div className="bg-blue-600/10 border border-blue-500/30 rounded-2xl p-6">
            <h4 className="text-blue-400 font-bold mb-3 flex items-center gap-2">
              ℹ️ How It Works
            </h4>
            <ul className="text-gray-300 text-sm space-y-2 list-disc list-inside">
              <li>Add managers and customize their FTD, deposits, partners, and salary data</li>
              <li>Data is stored locally in your browser (no backend needed)</li>
              <li>Edit or delete manager data at any time</li>
              <li>All dashboard screens will display the data you enter here</li>
              <li>Perfect for managing multiple affiliate managers</li>
            </ul>
          </div>
        </div>
      </div>
    </div>
  );
}