import { useState } from 'react';
import { useNavigate } from 'react-router';
import { UserPlus, Mail, Lock, Shield, Eye, EyeOff, User } from 'lucide-react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

export default function CreateAdminUser() {
  const navigate = useNavigate();
  const [showPassword, setShowPassword] = useState(false);
  const [formData, setFormData] = useState({
    fullName: '',
    email: '',
    password: '',
    confirmPassword: '',
    role: 'manager' as 'admin' | 'manager',
    nationality: '',
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    if (formData.password !== formData.confirmPassword) {
      alert('Passwords do not match!');
      return;
    }
    alert(`${formData.role === 'admin' ? 'Admin' : 'Manager'} account created successfully!`);
    navigate('/admin/dashboard');
  };

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 ml-64">
        <DashboardHeader title="Create New User" subtitle="Add a new manager or admin to the system" />
        
        <div className="p-8">
          <div className="max-w-3xl mx-auto">
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-8">
              <div className="flex items-center gap-3 mb-8">
                <div className="w-12 h-12 rounded-lg bg-purple-600/20 flex items-center justify-center">
                  <UserPlus className="w-6 h-6 text-purple-400" />
                </div>
                <div>
                  <h2 className="text-white text-2xl font-bold">Create User Account</h2>
                  <p className="text-gray-400">Set up a new admin or manager account</p>
                </div>
              </div>

              <form onSubmit={handleSubmit} className="space-y-6">
                {/* Role Selection */}
                <div>
                  <label className="block text-white mb-3 font-medium">Select Role *</label>
                  <div className="grid grid-cols-2 gap-4">
                    <button
                      type="button"
                      onClick={() => setFormData({ ...formData, role: 'manager' })}
                      className={`p-6 rounded-xl border-2 transition-all ${
                        formData.role === 'manager'
                          ? 'border-purple-500 bg-purple-500/20'
                          : 'border-gray-600 bg-gray-700/50 hover:border-gray-500'
                      }`}
                    >
                      <User className={`w-8 h-8 mx-auto mb-3 ${formData.role === 'manager' ? 'text-purple-400' : 'text-gray-400'}`} />
                      <p className={`font-semibold mb-1 ${formData.role === 'manager' ? 'text-white' : 'text-gray-400'}`}>
                        Manager
                      </p>
                      <p className="text-xs text-gray-500">Affiliate manager access</p>
                    </button>

                    <button
                      type="button"
                      onClick={() => setFormData({ ...formData, role: 'admin' })}
                      className={`p-6 rounded-xl border-2 transition-all ${
                        formData.role === 'admin'
                          ? 'border-blue-500 bg-blue-500/20'
                          : 'border-gray-600 bg-gray-700/50 hover:border-gray-500'
                      }`}
                    >
                      <Shield className={`w-8 h-8 mx-auto mb-3 ${formData.role === 'admin' ? 'text-blue-400' : 'text-gray-400'}`} />
                      <p className={`font-semibold mb-1 ${formData.role === 'admin' ? 'text-white' : 'text-gray-400'}`}>
                        Administrator
                      </p>
                      <p className="text-xs text-gray-500">Full system access</p>
                    </button>
                  </div>
                </div>

                {/* Full Name */}
                <div>
                  <label className="block text-white mb-2 font-medium">Full Name *</label>
                  <div className="relative">
                    <User className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                    <input
                      type="text"
                      value={formData.fullName}
                      onChange={(e) => setFormData({ ...formData, fullName: e.target.value })}
                      className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-4 py-3 text-white focus:border-purple-500 focus:outline-none"
                      placeholder="John Doe"
                      required
                    />
                  </div>
                </div>

                {/* Email */}
                <div>
                  <label className="block text-white mb-2 font-medium">Email Address *</label>
                  <div className="relative">
                    <Mail className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                    <input
                      type="email"
                      value={formData.email}
                      onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                      className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-4 py-3 text-white focus:border-purple-500 focus:outline-none"
                      placeholder="user@example.com"
                      required
                    />
                  </div>
                </div>

                {/* Nationality (for managers) */}
                {formData.role === 'manager' && (
                  <div>
                    <label className="block text-white mb-2 font-medium">Nationality *</label>
                    <select
                      value={formData.nationality}
                      onChange={(e) => setFormData({ ...formData, nationality: e.target.value })}
                      className="w-full bg-gray-700/50 border border-gray-600 rounded-xl px-4 py-3 text-white focus:border-purple-500 focus:outline-none"
                      required
                    >
                      <option value="">Select nationality</option>
                      <option value="usa">United States</option>
                      <option value="uk">United Kingdom</option>
                      <option value="canada">Canada</option>
                      <option value="australia">Australia</option>
                      <option value="germany">Germany</option>
                      <option value="singapore">Singapore</option>
                      <option value="japan">Japan</option>
                      <option value="other">Other</option>
                    </select>
                  </div>
                )}

                {/* Password */}
                <div>
                  <label className="block text-white mb-2 font-medium">Password *</label>
                  <div className="relative">
                    <Lock className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                    <input
                      type={showPassword ? 'text' : 'password'}
                      value={formData.password}
                      onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                      className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-12 py-3 text-white focus:border-purple-500 focus:outline-none"
                      placeholder="••••••••"
                      required
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      className="absolute right-4 top-1/2 -translate-y-1/2 text-gray-400 hover:text-white"
                    >
                      {showPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                    </button>
                  </div>
                </div>

                {/* Confirm Password */}
                <div>
                  <label className="block text-white mb-2 font-medium">Confirm Password *</label>
                  <div className="relative">
                    <Lock className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                    <input
                      type="password"
                      value={formData.confirmPassword}
                      onChange={(e) => setFormData({ ...formData, confirmPassword: e.target.value })}
                      className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-4 py-3 text-white focus:border-purple-500 focus:outline-none"
                      placeholder="••••••••"
                      required
                    />
                  </div>
                </div>

                {/* Info Box */}
                <div className="bg-blue-500/10 border border-blue-500/30 rounded-xl p-4">
                  <p className="text-blue-300 text-sm">
                    <strong>💡 Password Requirements:</strong> Minimum 8 characters with uppercase, lowercase, numbers, and special characters.
                  </p>
                </div>

                {/* Admin Permission Notice */}
                {formData.role === 'admin' && (
                  <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-xl p-4">
                    <p className="text-yellow-300 text-sm">
                      <strong>⚠️ Admin Role:</strong> This user will have full access to the system including user management, 
                      NDA approval, payroll processing, and all administrative functions.
                    </p>
                  </div>
                )}

                {/* Action Buttons */}
                <div className="flex gap-4 pt-6">
                  <button
                    type="button"
                    onClick={() => navigate('/admin/dashboard')}
                    className="flex-1 px-6 py-3 rounded-xl bg-gray-700 hover:bg-gray-600 text-white font-medium transition-colors"
                  >
                    Cancel
                  </button>
                  <button
                    type="submit"
                    className="flex-1 px-6 py-3 rounded-xl bg-gradient-to-r from-purple-600 to-purple-700 hover:from-purple-700 hover:to-purple-800 text-white font-semibold shadow-lg shadow-purple-600/50"
                  >
                    Create {formData.role === 'admin' ? 'Admin' : 'Manager'} Account
                  </button>
                </div>
              </form>
            </div>

            {/* Security Notice */}
            <div className="mt-6 bg-purple-500/10 border border-purple-500/30 rounded-xl p-4">
              <p className="text-purple-300 text-sm">
                <strong>🔒 Security Notice:</strong> The user will receive an email with login credentials. 
                They will be required to change their password on first login. All actions are logged for security auditing.
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
