import { useState } from 'react';
import { Upload, CheckCircle, AlertCircle, Download, RefreshCw, FileText } from 'lucide-react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

export default function ImportKPIData() {
  const [file, setFile] = useState<File | null>(null);
  const [importing, setImporting] = useState(false);
  const [result, setResult] = useState<any>(null);

  const handleFileUpload = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files && e.target.files[0]) {
      setFile(e.target.files[0]);
      setResult(null); // Clear previous results
    }
  };

  const processCSV = async () => {
    if (!file) return;
    
    setImporting(true);
    
    // Simulate CSV processing (in production, this would call your backend API)
    setTimeout(() => {
      setResult({
        success: true,
        message: 'Successfully imported 24 manager records',
        details: {
          processed: 24,
          updated: 24,
          failed: 0
        }
      });
      setImporting(false);
    }, 2000);
  };

  const downloadTemplate = () => {
    // Create CSV template
    const csvContent = `affiliate_email,ftd_count,deposits,partners,date
manager1@example.com,210,82000,35,2026-02-24
manager2@example.com,185,75000,30,2026-02-24
manager3@example.com,165,68000,28,2026-02-24`;
    
    const blob = new Blob([csvContent], { type: 'text/csv' });
    const url = window.URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = 'kpi_import_template.csv';
    a.click();
  };

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 ml-64">
        <DashboardHeader 
          title="Import KPI Data" 
          subtitle="Upload CSV data from affiliate platform" 
        />
        
        <div className="p-8">
          <div className="max-w-4xl mx-auto space-y-6">
            
            {/* Instructions Card */}
            <div className="bg-gradient-to-br from-blue-600/20 to-blue-800/20 backdrop-blur-lg border border-blue-500/30 rounded-2xl p-6">
              <div className="flex items-start gap-4">
                <div className="w-12 h-12 rounded-lg bg-blue-600/30 flex items-center justify-center flex-shrink-0">
                  <FileText className="w-6 h-6 text-blue-400" />
                </div>
                <div className="flex-1">
                  <h3 className="text-white text-xl font-bold mb-3">
                    📋 How to Import Data from Affiliate Platform
                  </h3>
                  <ol className="text-blue-200 text-sm space-y-2 list-decimal list-inside">
                    <li>Login to your <strong>7starswin affiliate platform</strong> (manager.7starswinpartners.com)</li>
                    <li>Navigate to <strong>Reports → Export Data</strong></li>
                    <li>Select date range and download CSV file</li>
                    <li>Upload the CSV file using the form below</li>
                    <li>Review and confirm the import</li>
                  </ol>
                </div>
              </div>
            </div>

            {/* CSV Format Reference */}
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-white font-bold text-lg">Required CSV Format</h3>
                <button
                  onClick={downloadTemplate}
                  className="flex items-center gap-2 px-4 py-2 rounded-lg bg-purple-600 hover:bg-purple-700 text-white text-sm font-medium transition-colors"
                >
                  <Download className="w-4 h-4" />
                  Download Template
                </button>
              </div>
              
              <div className="bg-gray-900/50 rounded-xl p-4 overflow-x-auto">
                <code className="text-emerald-400 text-xs font-mono block whitespace-pre">
{`affiliate_email,ftd_count,deposits,partners,date
manager1@example.com,210,82000,35,2026-02-24
manager2@example.com,185,75000,30,2026-02-24
manager3@example.com,165,68000,28,2026-02-24`}
                </code>
              </div>
              
              <div className="mt-4 grid md:grid-cols-2 gap-4">
                <div className="bg-purple-500/10 border border-purple-500/30 rounded-lg p-3">
                  <p className="text-purple-300 text-xs font-semibold mb-1">✅ Required Columns:</p>
                  <ul className="text-purple-200 text-xs space-y-1">
                    <li>• <strong>affiliate_email</strong> - Manager's email</li>
                    <li>• <strong>ftd_count</strong> - First Time Deposits</li>
                    <li>• <strong>deposits</strong> - Total deposit amount</li>
                    <li>• <strong>partners</strong> - Active partners count</li>
                    <li>• <strong>date</strong> - Date (YYYY-MM-DD)</li>
                  </ul>
                </div>
                
                <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-lg p-3">
                  <p className="text-yellow-300 text-xs font-semibold mb-1">⚠️ Important Notes:</p>
                  <ul className="text-yellow-200 text-xs space-y-1">
                    <li>• Email must match manager account</li>
                    <li>• Numbers should not include symbols</li>
                    <li>• Date format: YYYY-MM-DD</li>
                    <li>• File encoding: UTF-8</li>
                  </ul>
                </div>
              </div>
            </div>

            {/* Upload Section */}
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-8">
              <h3 className="text-white font-bold text-lg mb-6">Upload CSV File</h3>
              
              {/* File Upload Area */}
              <div className="mb-6">
                <div className="border-2 border-dashed border-gray-600 rounded-xl p-12 text-center hover:border-purple-500 hover:bg-purple-500/5 transition-all cursor-pointer">
                  <input
                    type="file"
                    accept=".csv"
                    onChange={handleFileUpload}
                    className="hidden"
                    id="csv-upload"
                  />
                  <label htmlFor="csv-upload" className="cursor-pointer">
                    <div className="w-16 h-16 rounded-full bg-purple-600/20 flex items-center justify-center mx-auto mb-4">
                      <Upload className="w-8 h-8 text-purple-400" />
                    </div>
                    <p className="text-white font-semibold text-lg mb-2">
                      {file ? file.name : 'Click to upload CSV file'}
                    </p>
                    <p className="text-gray-400 text-sm">
                      {file 
                        ? `File size: ${(file.size / 1024).toFixed(2)} KB` 
                        : 'Drag and drop or click to browse'}
                    </p>
                  </label>
                </div>
              </div>

              {/* Import Button */}
              <button
                onClick={processCSV}
                disabled={!file || importing}
                className="w-full bg-gradient-to-r from-purple-600 to-purple-700 hover:from-purple-700 hover:to-purple-800 disabled:opacity-50 disabled:cursor-not-allowed text-white py-4 rounded-xl font-semibold transition-all flex items-center justify-center gap-3 text-lg"
              >
                {importing ? (
                  <>
                    <RefreshCw className="w-5 h-5 animate-spin" />
                    Processing CSV...
                  </>
                ) : (
                  <>
                    <Upload className="w-5 h-5" />
                    Import Data to System
                  </>
                )}
              </button>

              {/* Result Display */}
              {result && (
                <div className={`mt-6 p-6 rounded-xl ${
                  result.success 
                    ? 'bg-emerald-500/10 border border-emerald-500/30' 
                    : 'bg-red-500/10 border border-red-500/30'
                }`}>
                  <div className="flex items-start gap-4">
                    {result.success ? (
                      <CheckCircle className="w-8 h-8 text-emerald-400 flex-shrink-0" />
                    ) : (
                      <AlertCircle className="w-8 h-8 text-red-400 flex-shrink-0" />
                    )}
                    <div className="flex-1">
                      <p className={`font-bold text-lg mb-2 ${
                        result.success ? 'text-emerald-300' : 'text-red-300'
                      }`}>
                        {result.success ? '✅ Import Successful!' : '❌ Import Failed'}
                      </p>
                      <p className="text-gray-300 text-sm mb-3">
                        {result.message}
                      </p>
                      
                      {result.details && (
                        <div className="grid grid-cols-3 gap-4 mt-4">
                          <div className="bg-gray-900/50 rounded-lg p-3">
                            <p className="text-gray-400 text-xs mb-1">Processed</p>
                            <p className="text-white font-bold text-xl">{result.details.processed}</p>
                          </div>
                          <div className="bg-gray-900/50 rounded-lg p-3">
                            <p className="text-gray-400 text-xs mb-1">Updated</p>
                            <p className="text-emerald-400 font-bold text-xl">{result.details.updated}</p>
                          </div>
                          <div className="bg-gray-900/50 rounded-lg p-3">
                            <p className="text-gray-400 text-xs mb-1">Failed</p>
                            <p className="text-red-400 font-bold text-xl">{result.details.failed}</p>
                          </div>
                        </div>
                      )}
                    </div>
                  </div>
                </div>
              )}
            </div>

            {/* Integration Options */}
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <h3 className="text-white font-bold text-lg mb-4">
                🔄 Automated Integration Options
              </h3>
              <div className="grid md:grid-cols-3 gap-4">
                <div className="bg-gray-900/50 rounded-xl p-4">
                  <div className="w-10 h-10 rounded-lg bg-blue-600/20 flex items-center justify-center mb-3">
                    <RefreshCw className="w-5 h-5 text-blue-400" />
                  </div>
                  <h4 className="text-white font-semibold mb-2">API Sync</h4>
                  <p className="text-gray-400 text-sm">
                    Connect via REST API for automatic updates every 30 minutes
                  </p>
                </div>
                
                <div className="bg-gray-900/50 rounded-xl p-4">
                  <div className="w-10 h-10 rounded-lg bg-purple-600/20 flex items-center justify-center mb-3">
                    <FileText className="w-5 h-5 text-purple-400" />
                  </div>
                  <h4 className="text-white font-semibold mb-2">Webhooks</h4>
                  <p className="text-gray-400 text-sm">
                    Real-time updates pushed from affiliate platform instantly
                  </p>
                </div>
                
                <div className="bg-gray-900/50 rounded-xl p-4">
                  <div className="w-10 h-10 rounded-lg bg-emerald-600/20 flex items-center justify-center mb-3">
                    <Download className="w-5 h-5 text-emerald-400" />
                  </div>
                  <h4 className="text-white font-semibold mb-2">CSV Import</h4>
                  <p className="text-gray-400 text-sm">
                    Manual daily/weekly imports from exported CSV files (current method)
                  </p>
                </div>
              </div>
              
              <div className="mt-4 bg-blue-500/10 border border-blue-500/30 rounded-xl p-4">
                <p className="text-blue-300 text-sm">
                  💡 <strong>Want automated sync?</strong> Contact your affiliate platform support 
                  to request API access or webhook integration. See <code className="text-blue-200">/AFFILIATE_INTEGRATION_GUIDE.md</code> for details.
                </p>
              </div>
            </div>

          </div>
        </div>
      </div>
    </div>
  );
}
