import { Link, useNavigate } from 'react-router';
import { useState } from 'react';
import { motion } from 'motion/react';
import { Lock, Mail, Shield, Eye, EyeOff } from 'lucide-react';
import { FootballAnimation } from '../components/3DAnimations';

interface LoginPageProps {
  isRegister?: boolean;
}

// Animated Galaxy Background Components
const GalaxyBackground = () => {
  // Generate random stars
  const stars = Array.from({ length: 150 }, (_, i) => ({
    id: i,
    x: Math.random() * 100,
    y: Math.random() * 100,
    size: Math.random() * 3 + 1,
    duration: Math.random() * 3 + 2,
    delay: Math.random() * 2,
  }));

  // Generate planets
  const planets = [
    { id: 1, x: 15, y: 20, size: 80, color: 'from-purple-500 to-pink-500', duration: 20 },
    { id: 2, x: 75, y: 70, size: 60, color: 'from-blue-500 to-cyan-500', duration: 25 },
    { id: 3, x: 85, y: 15, size: 40, color: 'from-orange-500 to-red-500', duration: 18 },
  ];

  return (
    <div className="absolute inset-0 overflow-hidden">
      {/* Cosmic nebula background */}
      <div className="absolute inset-0 bg-gradient-to-br from-indigo-950 via-purple-950 to-black" />
      
      {/* Animated nebula clouds */}
      <motion.div
        className="absolute top-0 left-0 w-full h-full opacity-30"
        animate={{
          background: [
            'radial-gradient(circle at 20% 30%, rgba(138, 43, 226, 0.4) 0%, transparent 50%)',
            'radial-gradient(circle at 80% 70%, rgba(138, 43, 226, 0.4) 0%, transparent 50%)',
            'radial-gradient(circle at 40% 60%, rgba(138, 43, 226, 0.4) 0%, transparent 50%)',
            'radial-gradient(circle at 20% 30%, rgba(138, 43, 226, 0.4) 0%, transparent 50%)',
          ],
        }}
        transition={{ duration: 20, repeat: Infinity, ease: 'linear' }}
      />

      {/* Sun - animated glowing orb */}
      <motion.div
        className="absolute"
        style={{ left: '10%', top: '10%' }}
        animate={{
          scale: [1, 1.2, 1],
          opacity: [0.8, 1, 0.8],
        }}
        transition={{ duration: 4, repeat: Infinity, ease: 'easeInOut' }}
      >
        <div className="relative w-32 h-32">
          {/* Sun glow */}
          <div className="absolute inset-0 rounded-full bg-gradient-to-br from-yellow-300 to-orange-500 blur-2xl opacity-60" />
          {/* Sun core */}
          <div className="absolute inset-4 rounded-full bg-gradient-to-br from-yellow-200 to-orange-400" />
          {/* Sun rays */}
          {[...Array(8)].map((_, i) => (
            <motion.div
              key={i}
              className="absolute top-1/2 left-1/2 w-1 h-16 bg-gradient-to-t from-yellow-400 to-transparent origin-bottom"
              style={{
                transform: `translate(-50%, -100%) rotate(${i * 45}deg)`,
              }}
              animate={{
                opacity: [0.3, 0.8, 0.3],
                scaleY: [0.8, 1.2, 0.8],
              }}
              transition={{
                duration: 2,
                repeat: Infinity,
                delay: i * 0.2,
              }}
            />
          ))}
        </div>
      </motion.div>

      {/* Moon - with craters */}
      <motion.div
        className="absolute"
        style={{ right: '8%', top: '25%' }}
        animate={{
          y: [0, -20, 0],
          rotate: [0, 5, 0],
        }}
        transition={{ duration: 8, repeat: Infinity, ease: 'easeInOut' }}
      >
        <div className="relative w-24 h-24">
          {/* Moon glow */}
          <div className="absolute inset-0 rounded-full bg-blue-300 blur-xl opacity-40" />
          {/* Moon surface */}
          <div className="absolute inset-2 rounded-full bg-gradient-to-br from-gray-200 to-gray-400 shadow-2xl">
            {/* Craters */}
            <div className="absolute top-4 left-4 w-3 h-3 rounded-full bg-gray-500 opacity-40" />
            <div className="absolute top-8 right-6 w-4 h-4 rounded-full bg-gray-500 opacity-30" />
            <div className="absolute bottom-6 left-8 w-2 h-2 rounded-full bg-gray-500 opacity-50" />
          </div>
        </div>
      </motion.div>

      {/* Planets with orbital motion */}
      {planets.map((planet) => (
        <motion.div
          key={planet.id}
          className="absolute"
          style={{ left: `${planet.x}%`, top: `${planet.y}%` }}
          animate={{
            y: [0, -30, 0],
            x: [0, 20, 0],
            rotate: [0, 360],
          }}
          transition={{
            duration: planet.duration,
            repeat: Infinity,
            ease: 'linear',
          }}
        >
          <div
            className={`rounded-full bg-gradient-to-br ${planet.color} shadow-2xl relative`}
            style={{ width: planet.size, height: planet.size }}
          >
            {/* Planet ring (for some planets) */}
            {planet.id === 2 && (
              <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[140%] h-4 border-4 border-cyan-400/30 rounded-full" 
                style={{ transform: 'translate(-50%, -50%) rotateX(75deg)' }}
              />
            )}
            {/* Planet glow */}
            <div className={`absolute inset-0 rounded-full bg-gradient-to-br ${planet.color} blur-lg opacity-40`} />
          </div>
        </motion.div>
      ))}

      {/* Twinkling stars */}
      {stars.map((star) => (
        <motion.div
          key={star.id}
          className="absolute rounded-full bg-white"
          style={{
            left: `${star.x}%`,
            top: `${star.y}%`,
            width: star.size,
            height: star.size,
          }}
          animate={{
            opacity: [0.2, 1, 0.2],
            scale: [0.8, 1.2, 0.8],
          }}
          transition={{
            duration: star.duration,
            repeat: Infinity,
            delay: star.delay,
          }}
        />
      ))}

      {/* Shooting stars */}
      {[1, 2, 3].map((i) => (
        <motion.div
          key={`shooting-${i}`}
          className="absolute w-1 h-1 bg-white rounded-full"
          initial={{ opacity: 0, x: '100vw', y: `${i * 30}%` }}
          animate={{
            opacity: [0, 1, 1, 0],
            x: ['-20%'],
            y: [`${i * 30 + 50}%`],
          }}
          transition={{
            duration: 2,
            repeat: Infinity,
            delay: i * 5,
            repeatDelay: 10,
          }}
        >
          <div className="w-20 h-0.5 bg-gradient-to-r from-white to-transparent" />
        </motion.div>
      ))}

      {/* Cosmic particles/dust */}
      {Array.from({ length: 30 }, (_, i) => ({
        id: i,
        x: Math.random() * 100,
        y: Math.random() * 100,
        size: Math.random() * 2 + 0.5,
      })).map((particle) => (
        <motion.div
          key={`particle-${particle.id}`}
          className="absolute rounded-full bg-purple-300"
          style={{
            left: `${particle.x}%`,
            top: `${particle.y}%`,
            width: particle.size,
            height: particle.size,
          }}
          animate={{
            y: [0, -100, 100, 0],
            opacity: [0.1, 0.3, 0.1],
          }}
          transition={{
            duration: 15 + particle.id,
            repeat: Infinity,
            ease: 'linear',
          }}
        />
      ))}
    </div>
  );
};

export default function LoginPage({ isRegister = false }: LoginPageProps) {
  const navigate = useNavigate();
  const [showPassword, setShowPassword] = useState(false);
  const [error, setError] = useState('');
  const [formData, setFormData] = useState({
    email: '',
    password: '',
    name: '',
    rememberMe: false,
  });

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    setError('');
    
    // Admin credentials - Only this admin account can login
    const ADMIN_EMAIL = 'luka@7starswinpartners.com';
    const ADMIN_PASSWORD = 'Luka2020.';
    
    const email = formData.email.toLowerCase().trim();
    const password = formData.password.trim();
    
    // Check for admin login
    if (email === ADMIN_EMAIL.toLowerCase() && password === ADMIN_PASSWORD) {
      navigate('/admin/dashboard');
      return;
    }
    
    // For managers: Check if account exists and is approved
    // In real app, this would query the database
    // For now, deny access to anyone not the admin
    setError('⚠️ Invalid credentials or access denied. Only approved users can login. Contact admin for assistance.');
  };

  return (
    <div className="min-h-screen relative flex items-center justify-center p-6 overflow-hidden">
      {/* Animated Galaxy Background */}
      <GalaxyBackground />
      
      {/* Content */}
      <div className="max-w-6xl w-full grid md:grid-cols-2 gap-8 items-center relative z-10">
        {/* Left Panel - 3D Animation */}
        <motion.div
          initial={{ opacity: 0, x: -50 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.8 }}
          className="hidden md:flex flex-col items-center justify-center bg-gradient-to-br from-purple-600/20 to-blue-600/20 backdrop-blur-lg border border-purple-500/30 rounded-3xl p-12"
        >
          <div className="mb-8">
            <div className="w-16 h-16 rounded-2xl bg-gradient-to-br from-purple-600 to-purple-800 flex items-center justify-center mb-4 mx-auto">
              <span className="text-white font-bold text-3xl">7★</span>
            </div>
            <h2 className="text-white text-3xl font-bold text-center mb-2">7starswin</h2>
            <p className="text-purple-300 text-center">Secure HR Management Platform</p>
          </div>
          
          <FootballAnimation />
          
          <div className="mt-8 space-y-4">
            <div className="flex items-center gap-3 text-purple-300">
              <Shield className="w-5 h-5" />
              <span className="text-sm">256-bit Encryption</span>
            </div>
            <div className="flex items-center gap-3 text-blue-300">
              <Lock className="w-5 h-5" />
              <span className="text-sm">Secure Authentication</span>
            </div>
          </div>
        </motion.div>

        {/* Right Panel - Form */}
        <motion.div
          initial={{ opacity: 0, x: 50 }}
          animate={{ opacity: 1, x: 0 }}
          transition={{ duration: 0.8 }}
          className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-3xl p-8 md:p-12"
        >
          <div className="mb-8">
            <h1 className="text-white text-3xl font-bold mb-2">
              Login to 7starswin
            </h1>
            <p className="text-gray-400">
              Admin and Manager portal access
            </p>
          </div>

          <form onSubmit={handleSubmit} className="space-y-6">
            {/* Email Field */}
            <div>
              <label className="block text-white mb-2 font-medium">Email Address</label>
              <div className="relative">
                <Mail className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                <input
                  type="email"
                  value={formData.email}
                  onChange={(e) => setFormData({ ...formData, email: e.target.value })}
                  className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-4 py-3 text-white focus:border-purple-500 focus:outline-none transition-colors"
                  placeholder="you@example.com"
                  required
                />
              </div>
            </div>

            {/* Password Field */}
            <div>
              <label className="block text-white mb-2 font-medium">Password</label>
              <div className="relative">
                <Lock className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                <input
                  type={showPassword ? 'text' : 'password'}
                  value={formData.password}
                  onChange={(e) => setFormData({ ...formData, password: e.target.value })}
                  className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-12 py-3 text-white focus:border-purple-500 focus:outline-none transition-colors"
                  placeholder="••••••••"
                  required
                />
                <button
                  type="button"
                  onClick={() => setShowPassword(!showPassword)}
                  className="absolute right-4 top-1/2 -translate-y-1/2 text-gray-400 hover:text-white transition-colors"
                >
                  {showPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                </button>
              </div>
            </div>

            {/* Remember Me & Forgot Password */}
            <div className="flex items-center justify-between">
              <label className="flex items-center gap-2 cursor-pointer">
                <input
                  type="checkbox"
                  checked={formData.rememberMe}
                  onChange={(e) => setFormData({ ...formData, rememberMe: e.target.checked })}
                  className="w-4 h-4 rounded border-gray-600 bg-gray-700 text-purple-600 focus:ring-purple-500"
                />
                <span className="text-gray-400 text-sm">Remember me</span>
              </label>
              <Link to="#" className="text-purple-400 hover:text-purple-300 text-sm transition-colors">
                Forgot password?
              </Link>
            </div>

            {/* Error Message */}
            {error && (
              <motion.div
                initial={{ opacity: 0, y: -10 }}
                animate={{ opacity: 1, y: 0 }}
                className="bg-red-500/10 border border-red-500/30 rounded-xl p-4"
              >
                <p className="text-red-300 text-sm">{error}</p>
              </motion.div>
            )}

            {/* Submit Button */}
            <button
              type="submit"
              className="w-full bg-gradient-to-r from-purple-600 to-purple-700 hover:from-purple-700 hover:to-purple-800 text-white py-3 rounded-xl font-semibold transition-all shadow-lg shadow-purple-600/50"
            >
              Sign In
            </button>

            {/* Security Badge */}
            <div className="flex items-center justify-center gap-2 text-emerald-400 text-sm">
              <Shield className="w-4 h-4" />
              <span>Secure Login - 256-bit Encrypted</span>
            </div>

            {/* Back to Home */}
            <p className="text-center">
              <Link to="/" className="text-gray-400 hover:text-white text-sm transition-colors">
                ← Back to Home
              </Link>
            </p>

            {/* Important Notice */}
            <div className="bg-blue-500/10 border border-blue-500/30 rounded-xl p-4 mt-6">
              <p className="text-blue-300 text-sm">
                <strong>⚠️ Important:</strong> Managers can ONLY login after admin approval. New applicants must submit an NDA request and wait for admin verification before receiving login credentials.
              </p>
            </div>
          </form>
        </motion.div>
      </div>
    </div>
  );
}