import { useState } from 'react';
import { useParams } from 'react-router';
import { User, TrendingUp, DollarSign, FileText, Activity, Calendar } from 'lucide-react';
import { LineChart, Line, BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from 'recharts';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

export default function ManagerProfile() {
  const { id } = useParams();
  const [activeTab, setActiveTab] = useState('overview');

  const manager = {
    name: 'Michael Chen',
    email: 'michael.chen@example.com',
    nationality: 'Singapore',
    dateOfBirth: '1990-05-15',
    joinedDate: '2026-01-15',
    status: 'active',
    salary: 6000,
  };

  const kpiData = [
    { month: 'Jan', ftd: 120, partners: 20, deposits: 45000 },
    { month: 'Feb', ftd: 150, partners: 25, deposits: 55000 },
    { month: 'Mar', ftd: 180, partners: 28, deposits: 68000 },
    { month: 'Apr', ftd: 165, partners: 30, deposits: 62000 },
    { month: 'May', ftd: 195, partners: 32, deposits: 75000 },
    { month: 'Jun', ftd: 210, partners: 35, deposits: 82000 },
  ];

  const activityLogs = [
    { date: '2026-02-24 10:30', action: 'Logged in to dashboard', type: 'login' },
    { date: '2026-02-24 10:35', action: 'Viewed KPI report', type: 'view' },
    { date: '2026-02-23 15:20', action: 'Updated salary information', type: 'update' },
    { date: '2026-02-22 09:15', action: 'Downloaded NDA document', type: 'download' },
    { date: '2026-02-21 14:45', action: 'Logged in to dashboard', type: 'login' },
  ];

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 ml-64">
        <DashboardHeader title="Manager Profile" subtitle="Detailed manager information and analytics" />
        
        <div className="p-8 space-y-6">
          {/* Profile Header */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-start justify-between">
              <div className="flex items-center gap-6">
                <div className="w-24 h-24 rounded-2xl bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center">
                  <span className="text-white font-bold text-3xl">
                    {manager.name.split(' ').map(n => n[0]).join('')}
                  </span>
                </div>
                <div>
                  <h2 className="text-white text-2xl font-bold mb-1">{manager.name}</h2>
                  <p className="text-gray-400 mb-3">{manager.email}</p>
                  <div className="flex items-center gap-4">
                    <span className="px-3 py-1 rounded-lg bg-emerald-500/20 text-emerald-400 text-sm font-medium">
                      Active
                    </span>
                    <span className="text-gray-400 text-sm">Joined: {manager.joinedDate}</span>
                  </div>
                </div>
              </div>
              <div className="flex gap-3">
                <button className="px-4 py-2 rounded-lg bg-purple-600 hover:bg-purple-700 text-white text-sm font-medium transition-colors">
                  Edit Profile
                </button>
                <button className="px-4 py-2 rounded-lg bg-emerald-600 hover:bg-emerald-700 text-white text-sm font-medium transition-colors">
                  Process Payout
                </button>
              </div>
            </div>
          </div>

          {/* Tabs */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl overflow-hidden">
            <div className="flex border-b border-gray-700">
              {[
                { id: 'overview', label: 'Overview', icon: User },
                { id: 'kpi', label: 'KPI Performance', icon: TrendingUp },
                { id: 'salary', label: 'Salary & Payout', icon: DollarSign },
                { id: 'documents', label: 'Documents', icon: FileText },
                { id: 'activity', label: 'Activity Logs', icon: Activity },
              ].map((tab) => {
                const Icon = tab.icon;
                return (
                  <button
                    key={tab.id}
                    onClick={() => setActiveTab(tab.id)}
                    className={`flex-1 flex items-center justify-center gap-2 px-6 py-4 transition-colors ${
                      activeTab === tab.id
                        ? 'bg-purple-600 text-white'
                        : 'text-gray-400 hover:text-white hover:bg-gray-700/50'
                    }`}
                  >
                    <Icon className="w-5 h-5" />
                    <span className="font-medium">{tab.label}</span>
                  </button>
                );
              })}
            </div>

            <div className="p-6">
              {/* Overview Tab */}
              {activeTab === 'overview' && (
                <div className="space-y-6">
                  <div className="grid md:grid-cols-2 gap-6">
                    <div className="space-y-4">
                      <h3 className="text-white text-lg font-bold mb-4">Personal Information</h3>
                      <div className="grid grid-cols-2 gap-4">
                        <div className="p-4 bg-gray-900/50 rounded-xl">
                          <p className="text-gray-400 text-sm mb-1">Full Name</p>
                          <p className="text-white font-medium">{manager.name}</p>
                        </div>
                        <div className="p-4 bg-gray-900/50 rounded-xl">
                          <p className="text-gray-400 text-sm mb-1">Nationality</p>
                          <p className="text-white font-medium">{manager.nationality}</p>
                        </div>
                        <div className="p-4 bg-gray-900/50 rounded-xl">
                          <p className="text-gray-400 text-sm mb-1">Date of Birth</p>
                          <p className="text-white font-medium">{manager.dateOfBirth}</p>
                        </div>
                        <div className="p-4 bg-gray-900/50 rounded-xl">
                          <p className="text-gray-400 text-sm mb-1">Email</p>
                          <p className="text-white font-medium text-sm">{manager.email}</p>
                        </div>
                      </div>
                    </div>

                    <div className="space-y-4">
                      <h3 className="text-white text-lg font-bold mb-4">Current Performance</h3>
                      <div className="grid grid-cols-2 gap-4">
                        <div className="p-4 bg-gradient-to-br from-blue-600/20 to-blue-800/20 border border-blue-500/30 rounded-xl">
                          <p className="text-gray-400 text-sm mb-1">Total FTD</p>
                          <p className="text-white text-2xl font-bold">210</p>
                        </div>
                        <div className="p-4 bg-gradient-to-br from-purple-600/20 to-purple-800/20 border border-purple-500/30 rounded-xl">
                          <p className="text-gray-400 text-sm mb-1">Partners</p>
                          <p className="text-white text-2xl font-bold">35</p>
                        </div>
                        <div className="p-4 bg-gradient-to-br from-emerald-600/20 to-emerald-800/20 border border-emerald-500/30 rounded-xl col-span-2">
                          <p className="text-gray-400 text-sm mb-1">Total Deposits</p>
                          <p className="text-white text-3xl font-bold">$82,000</p>
                        </div>
                      </div>
                    </div>
                  </div>
                </div>
              )}

              {/* KPI Tab */}
              {activeTab === 'kpi' && (
                <div className="space-y-6">
                  <h3 className="text-white text-lg font-bold">Performance Charts</h3>
                  <div className="bg-gray-900/50 rounded-xl p-6">
                    <h4 className="text-white font-semibold mb-4">FTD Trend</h4>
                    <ResponsiveContainer width="100%" height={300}>
                      <LineChart data={kpiData}>
                        <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                        <XAxis dataKey="month" stroke="#9CA3AF" />
                        <YAxis stroke="#9CA3AF" />
                        <Tooltip
                          contentStyle={{
                            backgroundColor: '#1F2937',
                            border: '1px solid #374151',
                            borderRadius: '0.5rem',
                          }}
                        />
                        <Line type="monotone" dataKey="ftd" stroke="#8B5CF6" strokeWidth={3} />
                      </LineChart>
                    </ResponsiveContainer>
                  </div>
                  <div className="bg-gray-900/50 rounded-xl p-6">
                    <h4 className="text-white font-semibold mb-4">Deposits Overview</h4>
                    <ResponsiveContainer width="100%" height={300}>
                      <BarChart data={kpiData}>
                        <CartesianGrid strokeDasharray="3 3" stroke="#374151" />
                        <XAxis dataKey="month" stroke="#9CA3AF" />
                        <YAxis stroke="#9CA3AF" />
                        <Tooltip
                          contentStyle={{
                            backgroundColor: '#1F2937',
                            border: '1px solid #374151',
                            borderRadius: '0.5rem',
                          }}
                        />
                        <Bar dataKey="deposits" fill="#10B981" radius={[8, 8, 0, 0]} />
                      </BarChart>
                    </ResponsiveContainer>
                  </div>
                </div>
              )}

              {/* Salary Tab */}
              {activeTab === 'salary' && (
                <div className="space-y-6">
                  <div className="grid md:grid-cols-3 gap-6">
                    <div className="bg-gradient-to-br from-emerald-600/20 to-emerald-800/20 border border-emerald-500/30 rounded-xl p-6">
                      <p className="text-gray-400 text-sm mb-2">Current Salary</p>
                      <p className="text-white text-3xl font-bold mb-1">${manager.salary.toLocaleString()}</p>
                      <p className="text-gray-400 text-sm">Per month</p>
                    </div>
                    <div className="bg-gray-900/50 rounded-xl p-6">
                      <p className="text-gray-400 text-sm mb-2">Last Payment</p>
                      <p className="text-white text-2xl font-bold mb-1">Jan 31, 2026</p>
                      <p className="text-emerald-400 text-sm">Paid successfully</p>
                    </div>
                    <div className="bg-gray-900/50 rounded-xl p-6">
                      <p className="text-gray-400 text-sm mb-2">Next Payment</p>
                      <p className="text-white text-2xl font-bold mb-1">Feb 28, 2026</p>
                      <p className="text-yellow-400 text-sm">Scheduled</p>
                    </div>
                  </div>
                  <div className="bg-gray-900/50 rounded-xl p-6">
                    <h4 className="text-white font-semibold mb-4">Payment History</h4>
                    <div className="space-y-3">
                      {[
                        { month: 'January 2026', amount: 6000, status: 'paid', date: 'Jan 31, 2026' },
                        { month: 'December 2025', amount: 6000, status: 'paid', date: 'Dec 31, 2025' },
                        { month: 'November 2025', amount: 5500, status: 'paid', date: 'Nov 30, 2025' },
                      ].map((payment, index) => (
                        <div key={index} className="flex items-center justify-between p-4 bg-gray-800/50 rounded-lg">
                          <div>
                            <p className="text-white font-medium">{payment.month}</p>
                            <p className="text-gray-400 text-sm">{payment.date}</p>
                          </div>
                          <div className="text-right">
                            <p className="text-emerald-400 font-bold">${payment.amount.toLocaleString()}</p>
                            <span className="text-emerald-400 text-xs">Paid</span>
                          </div>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>
              )}

              {/* Documents Tab */}
              {activeTab === 'documents' && (
                <div className="space-y-6">
                  <h3 className="text-white text-lg font-bold">Manager Documents</h3>
                  <div className="grid md:grid-cols-2 gap-4">
                    {[
                      { name: 'NDA Document', type: 'NDA', date: '2026-02-24', status: 'approved' },
                      { name: 'Passport Front', type: 'ID', date: '2026-02-20', status: 'verified' },
                      { name: 'Passport Back', type: 'ID', date: '2026-02-20', status: 'verified' },
                      { name: 'Bank Details', type: 'Payment', date: '2026-02-22', status: 'verified' },
                    ].map((doc, index) => (
                      <div key={index} className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                        <div className="flex items-center gap-3">
                          <div className="w-12 h-12 rounded-lg bg-purple-600/20 flex items-center justify-center">
                            <FileText className="w-6 h-6 text-purple-400" />
                          </div>
                          <div>
                            <p className="text-white font-medium">{doc.name}</p>
                            <p className="text-gray-400 text-sm">{doc.type} • {doc.date}</p>
                          </div>
                        </div>
                        <span className="px-3 py-1 rounded-lg bg-emerald-500/20 text-emerald-400 text-xs font-medium">
                          {doc.status}
                        </span>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Activity Tab */}
              {activeTab === 'activity' && (
                <div className="space-y-6">
                  <h3 className="text-white text-lg font-bold">Activity Timeline</h3>
                  <div className="space-y-3">
                    {activityLogs.map((log, index) => (
                      <div key={index} className="flex gap-4">
                        <div className="flex flex-col items-center">
                          <div className="w-3 h-3 rounded-full bg-purple-600"></div>
                          {index < activityLogs.length - 1 && (
                            <div className="w-0.5 h-full bg-gray-700 mt-2"></div>
                          )}
                        </div>
                        <div className="flex-1 pb-6">
                          <div className="p-4 bg-gray-900/50 rounded-xl">
                            <p className="text-white font-medium">{log.action}</p>
                            <div className="flex items-center gap-2 mt-2">
                              <Calendar className="w-3 h-3 text-gray-500" />
                              <p className="text-gray-500 text-xs">{log.date}</p>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
