import { useState } from 'react';
import { useParams, useNavigate } from 'react-router';
import { CheckCircle, XCircle, FileText, User, TrendingUp, DollarSign, Calendar, Flag, Shield } from 'lucide-react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

export default function NDAReviewApproval() {
  const { id } = useParams();
  const navigate = useNavigate();
  const [decision, setDecision] = useState<'approve' | 'reject' | null>(null);
  const [signature, setSignature] = useState('');
  const [notes, setNotes] = useState('');

  // Mock data
  const applicant = {
    id: '1',
    name: 'Michael Chen',
    nationality: 'Singapore',
    dateOfBirth: '1990-05-15',
    email: 'michael.chen@example.com',
    appliedDate: '2026-02-20',
    status: 'pending',
    kpi: {
      ftd: 150,
      partners: 25,
      totalDeposit: 75000,
      prohibitionPeriod: 90,
    },
    expectedSalary: 6000,
    documents: [
      { name: 'passport_front.pdf', size: 2.4, uploadDate: '2026-02-20' },
      { name: 'passport_back.pdf', size: 2.1, uploadDate: '2026-02-20' },
    ],
  };

  const auditLog = [
    { date: '2026-02-20 14:30', action: 'Application submitted', by: 'Michael Chen' },
    { date: '2026-02-20 14:35', action: 'Documents uploaded', by: 'Michael Chen' },
    { date: '2026-02-21 09:15', action: 'Under review', by: 'Admin - John Doe' },
  ];

  const handleApprove = () => {
    if (!signature) {
      alert('Please provide your digital signature');
      return;
    }
    // Navigate to NDA document
    navigate(`/nda-document/${id}`);
  };

  const handleReject = () => {
    if (!notes) {
      alert('Please provide rejection reason');
      return;
    }
    navigate('/admin/dashboard');
  };

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 ml-64">
        <DashboardHeader title="NDA Review & Approval" subtitle="Review and approve NDA applications" />
        
        <div className="p-8 space-y-6">
          {/* Applicant Profile Card */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-start justify-between mb-6">
              <div className="flex items-center gap-4">
                <div className="w-16 h-16 rounded-full bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center">
                  <span className="text-white font-bold text-xl">
                    {applicant.name.split(' ').map(n => n[0]).join('')}
                  </span>
                </div>
                <div>
                  <h2 className="text-white text-2xl font-bold">{applicant.name}</h2>
                  <p className="text-gray-400">{applicant.email}</p>
                  <div className="flex items-center gap-4 mt-2">
                    <div className="flex items-center gap-1 text-gray-400 text-sm">
                      <Flag className="w-4 h-4" />
                      {applicant.nationality}
                    </div>
                    <div className="flex items-center gap-1 text-gray-400 text-sm">
                      <Calendar className="w-4 h-4" />
                      Born: {applicant.dateOfBirth}
                    </div>
                  </div>
                </div>
              </div>
              <div className="text-right">
                <span className="px-4 py-2 rounded-lg bg-yellow-500/20 text-yellow-400 text-sm font-medium">
                  Pending Review
                </span>
                <p className="text-gray-400 text-sm mt-2">Applied: {applicant.appliedDate}</p>
              </div>
            </div>

            {/* Personal Info Grid */}
            <div className="grid md:grid-cols-3 gap-4 mt-6 p-4 bg-gray-900/50 rounded-xl">
              <div>
                <p className="text-gray-400 text-sm mb-1">Nationality</p>
                <p className="text-white font-medium">{applicant.nationality}</p>
              </div>
              <div>
                <p className="text-gray-400 text-sm mb-1">Date of Birth</p>
                <p className="text-white font-medium">{applicant.dateOfBirth}</p>
              </div>
              <div>
                <p className="text-gray-400 text-sm mb-1">Email</p>
                <p className="text-white font-medium">{applicant.email}</p>
              </div>
            </div>
          </div>

          <div className="grid md:grid-cols-2 gap-6">
            {/* KPI Summary */}
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <div className="flex items-center gap-3 mb-6">
                <div className="w-10 h-10 rounded-lg bg-blue-600/20 flex items-center justify-center">
                  <TrendingUp className="w-5 h-5 text-blue-400" />
                </div>
                <h3 className="text-white text-xl font-bold">KPI Summary</h3>
              </div>

              <div className="space-y-4">
                <div className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                  <div>
                    <p className="text-gray-400 text-sm">First Time Deposits</p>
                    <p className="text-white text-2xl font-bold">{applicant.kpi.ftd}</p>
                  </div>
                  <div className="w-12 h-12 rounded-lg bg-blue-600/20 flex items-center justify-center">
                    <TrendingUp className="w-6 h-6 text-blue-400" />
                  </div>
                </div>

                <div className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                  <div>
                    <p className="text-gray-400 text-sm">Partners</p>
                    <p className="text-white text-2xl font-bold">{applicant.kpi.partners}</p>
                  </div>
                  <div className="w-12 h-12 rounded-lg bg-purple-600/20 flex items-center justify-center">
                    <User className="w-6 h-6 text-purple-400" />
                  </div>
                </div>

                <div className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                  <div>
                    <p className="text-gray-400 text-sm">Total Deposit</p>
                    <p className="text-white text-2xl font-bold">${applicant.kpi.totalDeposit.toLocaleString()}</p>
                  </div>
                  <div className="w-12 h-12 rounded-lg bg-emerald-600/20 flex items-center justify-center">
                    <DollarSign className="w-6 h-6 text-emerald-400" />
                  </div>
                </div>

                <div className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                  <div>
                    <p className="text-gray-400 text-sm">Prohibition Period</p>
                    <p className="text-white text-2xl font-bold">{applicant.kpi.prohibitionPeriod} days</p>
                  </div>
                  <div className="w-12 h-12 rounded-lg bg-amber-600/20 flex items-center justify-center">
                    <Calendar className="w-6 h-6 text-amber-400" />
                  </div>
                </div>
              </div>
            </div>

            {/* Salary & Documents */}
            <div className="space-y-6">
              {/* Expected Salary */}
              <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
                <div className="flex items-center gap-3 mb-4">
                  <div className="w-10 h-10 rounded-lg bg-emerald-600/20 flex items-center justify-center">
                    <DollarSign className="w-5 h-5 text-emerald-400" />
                  </div>
                  <h3 className="text-white text-xl font-bold">Salary Expectation</h3>
                </div>
                <p className="text-white text-3xl font-bold">${applicant.expectedSalary.toLocaleString()}/mo</p>
                <p className="text-gray-400 text-sm mt-2">After prohibition period</p>
              </div>

              {/* Documents */}
              <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
                <div className="flex items-center gap-3 mb-4">
                  <div className="w-10 h-10 rounded-lg bg-amber-600/20 flex items-center justify-center">
                    <FileText className="w-5 h-5 text-amber-400" />
                  </div>
                  <h3 className="text-white text-xl font-bold">Documents</h3>
                </div>
                <div className="space-y-2">
                  {applicant.documents.map((doc, index) => (
                    <div key={index} className="flex items-center justify-between p-3 bg-gray-900/50 rounded-lg">
                      <div className="flex items-center gap-3">
                        <FileText className="w-5 h-5 text-gray-400" />
                        <div>
                          <p className="text-white text-sm font-medium">{doc.name}</p>
                          <p className="text-gray-400 text-xs">{doc.size} MB</p>
                        </div>
                      </div>
                      <button className="text-purple-400 hover:text-purple-300 text-sm font-medium">
                        View
                      </button>
                    </div>
                  ))}
                </div>
              </div>
            </div>
          </div>

          {/* Audit Log Timeline */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <h3 className="text-white text-xl font-bold mb-6">Audit Log Timeline</h3>
            <div className="space-y-4">
              {auditLog.map((log, index) => (
                <div key={index} className="flex gap-4">
                  <div className="flex flex-col items-center">
                    <div className="w-3 h-3 rounded-full bg-purple-600"></div>
                    {index < auditLog.length - 1 && (
                      <div className="w-0.5 h-full bg-gray-700 mt-2"></div>
                    )}
                  </div>
                  <div className="flex-1 pb-6">
                    <p className="text-white font-medium">{log.action}</p>
                    <p className="text-gray-400 text-sm">{log.by}</p>
                    <p className="text-gray-500 text-xs mt-1">{log.date}</p>
                  </div>
                </div>
              ))}
            </div>
          </div>

          {/* Decision Section */}
          {!decision && (
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <h3 className="text-white text-xl font-bold mb-6">Make Decision</h3>
              <div className="grid md:grid-cols-2 gap-4 mb-6">
                <button
                  onClick={() => setDecision('approve')}
                  className="p-6 rounded-xl border-2 border-emerald-600 bg-emerald-600/10 hover:bg-emerald-600/20 transition-all text-center"
                >
                  <CheckCircle className="w-12 h-12 text-emerald-400 mx-auto mb-3" />
                  <h4 className="text-white font-semibold text-lg mb-2">Approve Application</h4>
                  <p className="text-gray-400 text-sm">
                    Generate NDA and proceed with onboarding
                  </p>
                </button>
                <button
                  onClick={() => setDecision('reject')}
                  className="p-6 rounded-xl border-2 border-red-600 bg-red-600/10 hover:bg-red-600/20 transition-all text-center"
                >
                  <XCircle className="w-12 h-12 text-red-400 mx-auto mb-3" />
                  <h4 className="text-white font-semibold text-lg mb-2">Reject Application</h4>
                  <p className="text-gray-400 text-sm">
                    Decline the application with reason
                  </p>
                </button>
              </div>
            </div>
          )}

          {/* Approve Form */}
          {decision === 'approve' && (
            <div className="bg-gray-800/50 backdrop-blur-lg border border-emerald-600 rounded-2xl p-6">
              <h3 className="text-white text-xl font-bold mb-6 flex items-center gap-2">
                <CheckCircle className="w-6 h-6 text-emerald-400" />
                Approve Application
              </h3>
              <div className="space-y-4">
                <div>
                  <label className="block text-white mb-2 font-medium">Digital Signature</label>
                  <input
                    type="text"
                    value={signature}
                    onChange={(e) => setSignature(e.target.value)}
                    className="w-full bg-gray-700/50 border border-gray-600 rounded-xl px-4 py-3 text-white focus:border-emerald-500 focus:outline-none"
                    placeholder="Enter your full name as digital signature"
                  />
                </div>
                <div>
                  <label className="block text-white mb-2 font-medium">Notes (Optional)</label>
                  <textarea
                    value={notes}
                    onChange={(e) => setNotes(e.target.value)}
                    className="w-full bg-gray-700/50 border border-gray-600 rounded-xl px-4 py-3 text-white focus:border-emerald-500 focus:outline-none h-24"
                    placeholder="Add any internal notes..."
                  />
                </div>
                <div className="bg-blue-500/10 border border-blue-500/30 rounded-xl p-4">
                  <p className="text-blue-300 text-sm flex items-center gap-2">
                    <Shield className="w-4 h-4" />
                    Your approval will generate the NDA document and notify the applicant via email.
                  </p>
                </div>
                <div className="flex gap-4">
                  <button
                    onClick={() => setDecision(null)}
                    className="px-6 py-3 rounded-xl bg-gray-700 hover:bg-gray-600 text-white font-medium"
                  >
                    Cancel
                  </button>
                  <button
                    onClick={handleApprove}
                    className="flex-1 px-6 py-3 rounded-xl bg-gradient-to-r from-emerald-600 to-emerald-700 hover:from-emerald-700 hover:to-emerald-800 text-white font-semibold shadow-lg shadow-emerald-600/50"
                  >
                    Confirm Approval
                  </button>
                </div>
              </div>
            </div>
          )}

          {/* Reject Form */}
          {decision === 'reject' && (
            <div className="bg-gray-800/50 backdrop-blur-lg border border-red-600 rounded-2xl p-6">
              <h3 className="text-white text-xl font-bold mb-6 flex items-center gap-2">
                <XCircle className="w-6 h-6 text-red-400" />
                Reject Application
              </h3>
              <div className="space-y-4">
                <div>
                  <label className="block text-white mb-2 font-medium">Rejection Reason</label>
                  <textarea
                    value={notes}
                    onChange={(e) => setNotes(e.target.value)}
                    className="w-full bg-gray-700/50 border border-gray-600 rounded-xl px-4 py-3 text-white focus:border-red-500 focus:outline-none h-32"
                    placeholder="Explain why this application is being rejected..."
                    required
                  />
                </div>
                <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-xl p-4">
                  <p className="text-yellow-300 text-sm">
                    ⚠️ The applicant will be notified via email with your rejection reason.
                  </p>
                </div>
                <div className="flex gap-4">
                  <button
                    onClick={() => setDecision(null)}
                    className="px-6 py-3 rounded-xl bg-gray-700 hover:bg-gray-600 text-white font-medium"
                  >
                    Cancel
                  </button>
                  <button
                    onClick={handleReject}
                    className="flex-1 px-6 py-3 rounded-xl bg-gradient-to-r from-red-600 to-red-700 hover:from-red-700 hover:to-red-800 text-white font-semibold shadow-lg shadow-red-600/50"
                  >
                    Confirm Rejection
                  </button>
                </div>
              </div>
            </div>
          )}

          {/* Legal Disclaimer */}
          <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-xl p-4">
            <p className="text-yellow-400 text-sm">
              <strong>⚠️ Legal Disclaimer:</strong> This approval is for administrative purposes only. 
              All NDA documents should be reviewed by qualified legal counsel. By approving, you confirm 
              that proper due diligence has been conducted.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
