import { Link } from 'react-router';
import { motion } from 'motion/react';
import { CheckCircle, Clock, Mail, Shield, Home, FileCheck } from 'lucide-react';
import LegalFooter from '../components/LegalFooter';

// Animated Galaxy Background Components (matching LoginPage)
const GalaxyBackground = () => {
  // Generate random stars
  const stars = Array.from({ length: 200 }, (_, i) => ({
    id: i,
    x: Math.random() * 100,
    y: Math.random() * 100,
    size: Math.random() * 3 + 1,
    duration: Math.random() * 3 + 2,
    delay: Math.random() * 2,
  }));

  // Generate planets
  const planets = [
    { id: 1, x: 10, y: 15, size: 90, color: 'from-emerald-500 to-teal-500', duration: 22 },
    { id: 2, x: 80, y: 60, size: 70, color: 'from-purple-500 to-pink-500', duration: 28 },
    { id: 3, x: 75, y: 20, size: 50, color: 'from-blue-400 to-cyan-400', duration: 20 },
    { id: 4, x: 20, y: 75, size: 45, color: 'from-orange-400 to-red-500', duration: 25 },
  ];

  // Generate comets
  const comets = [
    { id: 1, delay: 3 },
    { id: 2, delay: 8 },
    { id: 3, delay: 13 },
    { id: 4, delay: 18 },
  ];

  return (
    <div className="fixed inset-0 overflow-hidden pointer-events-none">
      {/* Deep space background */}
      <div className="absolute inset-0 bg-gradient-to-br from-indigo-950 via-purple-950 to-black" />
      
      {/* Animated nebula clouds */}
      <motion.div
        className="absolute top-0 left-0 w-full h-full opacity-40"
        animate={{
          background: [
            'radial-gradient(circle at 25% 35%, rgba(138, 43, 226, 0.5) 0%, transparent 50%)',
            'radial-gradient(circle at 75% 65%, rgba(16, 185, 129, 0.4) 0%, transparent 50%)',
            'radial-gradient(circle at 50% 50%, rgba(59, 130, 246, 0.4) 0%, transparent 50%)',
            'radial-gradient(circle at 25% 35%, rgba(138, 43, 226, 0.5) 0%, transparent 50%)',
          ],
        }}
        transition={{ duration: 25, repeat: Infinity, ease: 'linear' }}
      />

      {/* Sun - glowing orb top left */}
      <motion.div
        className="absolute"
        style={{ left: '8%', top: '12%' }}
        animate={{
          scale: [1, 1.3, 1],
          opacity: [0.7, 1, 0.7],
        }}
        transition={{ duration: 5, repeat: Infinity, ease: 'easeInOut' }}
      >
        <div className="relative w-36 h-36">
          {/* Sun glow */}
          <div className="absolute inset-0 rounded-full bg-gradient-to-br from-yellow-200 to-orange-400 blur-3xl opacity-70" />
          {/* Sun core */}
          <div className="absolute inset-4 rounded-full bg-gradient-to-br from-yellow-100 to-orange-300" />
          {/* Sun rays - more dramatic */}
          {[...Array(12)].map((_, i) => (
            <motion.div
              key={i}
              className="absolute top-1/2 left-1/2 w-1.5 h-20 bg-gradient-to-t from-yellow-300 to-transparent origin-bottom"
              style={{
                transform: `translate(-50%, -100%) rotate(${i * 30}deg)`,
              }}
              animate={{
                opacity: [0.4, 1, 0.4],
                scaleY: [0.7, 1.4, 0.7],
              }}
              transition={{
                duration: 2.5,
                repeat: Infinity,
                delay: i * 0.15,
              }}
            />
          ))}
        </div>
      </motion.div>

      {/* Moon - with more detail */}
      <motion.div
        className="absolute"
        style={{ right: '10%', top: '20%' }}
        animate={{
          y: [0, -25, 0],
          rotate: [0, 8, 0],
        }}
        transition={{ duration: 10, repeat: Infinity, ease: 'easeInOut' }}
      >
        <div className="relative w-28 h-28">
          {/* Moon glow - stronger */}
          <div className="absolute inset-0 rounded-full bg-blue-200 blur-2xl opacity-50" />
          {/* Moon surface */}
          <div className="absolute inset-2 rounded-full bg-gradient-to-br from-gray-100 to-gray-400 shadow-2xl">
            {/* More craters */}
            <div className="absolute top-3 left-4 w-4 h-4 rounded-full bg-gray-500 opacity-40" />
            <div className="absolute top-8 right-5 w-5 h-5 rounded-full bg-gray-500 opacity-35" />
            <div className="absolute bottom-5 left-7 w-3 h-3 rounded-full bg-gray-500 opacity-50" />
            <div className="absolute top-6 left-12 w-2 h-2 rounded-full bg-gray-600 opacity-45" />
          </div>
        </div>
      </motion.div>

      {/* Planets with orbital motion */}
      {planets.map((planet) => (
        <motion.div
          key={planet.id}
          className="absolute"
          style={{ left: `${planet.x}%`, top: `${planet.y}%` }}
          animate={{
            y: [0, -40, 0],
            x: [0, 25, 0],
            rotate: [0, 360],
          }}
          transition={{
            duration: planet.duration,
            repeat: Infinity,
            ease: 'linear',
          }}
        >
          <div
            className={`rounded-full bg-gradient-to-br ${planet.color} shadow-2xl relative`}
            style={{ width: planet.size, height: planet.size }}
          >
            {/* Planet rings */}
            {planet.id === 2 && (
              <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[150%] h-5 border-4 border-pink-400/40 rounded-full" 
                style={{ transform: 'translate(-50%, -50%) rotateX(75deg)' }}
              />
            )}
            {planet.id === 3 && (
              <div className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[160%] h-4 border-3 border-cyan-300/30 rounded-full" 
                style={{ transform: 'translate(-50%, -50%) rotateX(70deg)' }}
              />
            )}
            {/* Planet glow */}
            <div className={`absolute inset-0 rounded-full bg-gradient-to-br ${planet.color} blur-xl opacity-50`} />
            {/* Surface details */}
            <div className="absolute top-4 right-6 w-3 h-3 rounded-full bg-black/20" />
            <div className="absolute bottom-6 left-5 w-4 h-4 rounded-full bg-white/10" />
          </div>
        </motion.div>
      ))}

      {/* Twinkling stars - more stars! */}
      {stars.map((star) => (
        <motion.div
          key={star.id}
          className="absolute rounded-full bg-white"
          style={{
            left: `${star.x}%`,
            top: `${star.y}%`,
            width: star.size,
            height: star.size,
          }}
          animate={{
            opacity: [0.2, 1, 0.2],
            scale: [0.7, 1.3, 0.7],
          }}
          transition={{
            duration: star.duration,
            repeat: Infinity,
            delay: star.delay,
          }}
        />
      ))}

      {/* Multiple shooting stars/comets */}
      {comets.map((comet) => (
        <motion.div
          key={`comet-${comet.id}`}
          className="absolute w-1 h-1 bg-white rounded-full"
          initial={{ opacity: 0, x: '100vw', y: `${comet.id * 20}%` }}
          animate={{
            opacity: [0, 1, 1, 0],
            x: ['-20%'],
            y: [`${comet.id * 20 + 60}%`],
          }}
          transition={{
            duration: 2.5,
            repeat: Infinity,
            delay: comet.delay,
            repeatDelay: 8,
          }}
        >
          <div className="w-24 h-0.5 bg-gradient-to-r from-white via-blue-200 to-transparent" />
        </motion.div>
      ))}

      {/* Cosmic particles/dust - more particles */}
      {Array.from({ length: 50 }, (_, i) => ({
        id: i,
        x: Math.random() * 100,
        y: Math.random() * 100,
        size: Math.random() * 2.5 + 0.5,
      })).map((particle) => (
        <motion.div
          key={`particle-${particle.id}`}
          className="absolute rounded-full bg-purple-200"
          style={{
            left: `${particle.x}%`,
            top: `${particle.y}%`,
            width: particle.size,
            height: particle.size,
          }}
          animate={{
            y: [0, -120, 120, 0],
            opacity: [0.1, 0.4, 0.1],
          }}
          transition={{
            duration: 18 + particle.id,
            repeat: Infinity,
            ease: 'linear',
          }}
        />
      ))}

      {/* Distant galaxies - small spiral effects */}
      {[1, 2, 3].map((i) => (
        <motion.div
          key={`galaxy-${i}`}
          className="absolute"
          style={{ 
            left: `${20 + i * 25}%`, 
            top: `${30 + i * 15}%`,
          }}
          animate={{
            rotate: [0, 360],
            opacity: [0.2, 0.4, 0.2],
          }}
          transition={{
            duration: 40 + i * 10,
            repeat: Infinity,
            ease: 'linear',
          }}
        >
          <div className="relative w-16 h-16">
            <div className="absolute inset-0 rounded-full bg-gradient-radial from-purple-400/20 via-blue-400/10 to-transparent blur-md" />
            <div className="absolute inset-2 rounded-full bg-gradient-radial from-white/30 via-purple-300/20 to-transparent blur-sm" />
          </div>
        </motion.div>
      ))}
    </div>
  );
};

export default function NDASubmittedPage() {
  return (
    <div className="min-h-screen relative overflow-hidden">
      {/* Animated Galaxy Background */}
      <GalaxyBackground />

      {/* Content with higher z-index */}
      <div className="relative z-10">
        {/* Header */}
        <nav className="bg-gray-900/70 backdrop-blur-xl border-b border-purple-500/30">
          <div className="max-w-7xl mx-auto px-6 py-4 flex items-center justify-between">
            <Link to="/" className="flex items-center gap-3">
              <div className="w-10 h-10 rounded-lg bg-gradient-to-br from-purple-600 to-purple-800 flex items-center justify-center shadow-lg shadow-purple-500/50">
                <span className="text-white font-bold text-xl">7★</span>
              </div>
              <span className="text-white font-bold text-xl">7starswin</span>
            </Link>
          </div>
        </nav>

        <div className="max-w-3xl mx-auto px-6 py-20">
          <motion.div
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.5 }}
            className="bg-gray-800/60 backdrop-blur-xl border border-purple-500/30 rounded-3xl p-12 text-center shadow-2xl"
          >
            {/* Success Icon */}
            <motion.div
              initial={{ scale: 0 }}
              animate={{ scale: 1 }}
              transition={{ delay: 0.2, type: "spring", stiffness: 200 }}
              className="mb-8"
            >
              <div className="w-24 h-24 mx-auto rounded-full bg-emerald-500/20 border-4 border-emerald-500 flex items-center justify-center">
                <CheckCircle className="w-12 h-12 text-emerald-400" />
              </div>
            </motion.div>

            {/* Title */}
            <motion.h1
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.3 }}
              className="text-white text-4xl font-bold mb-4"
            >
              NDA Request Submitted Successfully!
            </motion.h1>

            <motion.p
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.4 }}
              className="text-gray-300 text-lg mb-12"
            >
              Thank you for applying to join the 7starswin affiliate network
            </motion.p>

            {/* Info Cards */}
            <div className="grid md:grid-cols-2 gap-6 mb-12">
              <motion.div
                initial={{ opacity: 0, x: -20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: 0.5 }}
                className="bg-purple-500/10 border border-purple-500/30 rounded-2xl p-6 text-left"
              >
                <Clock className="w-10 h-10 text-purple-400 mb-4" />
                <h3 className="text-white font-bold text-lg mb-2">What's Next?</h3>
                <p className="text-gray-400 text-sm">
                  Our admin team will review your application within <strong className="text-purple-300">24-48 hours</strong>. 
                  We'll verify your documents and KPI data.
                </p>
              </motion.div>

              <motion.div
                initial={{ opacity: 0, x: 20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: 0.6 }}
                className="bg-blue-500/10 border border-blue-500/30 rounded-2xl p-6 text-left"
              >
                <Mail className="w-10 h-10 text-blue-400 mb-4" />
                <h3 className="text-white font-bold text-lg mb-2">Email Notification</h3>
                <p className="text-gray-400 text-sm">
                  You'll receive an email once your application is <strong className="text-blue-300">approved or requires additional information</strong>.
                </p>
              </motion.div>
            </div>

            {/* Important Notice */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.7 }}
              className="bg-yellow-500/10 border border-yellow-500/30 rounded-2xl p-6 mb-8"
            >
              <div className="flex items-start gap-4">
                <Shield className="w-6 h-6 text-yellow-400 mt-1 flex-shrink-0" />
                <div className="text-left">
                  <h3 className="text-yellow-300 font-bold mb-2">⚠️ Important: Login Access</h3>
                  <p className="text-gray-300 text-sm">
                    <strong>You cannot login yet.</strong> Your account will be activated only after admin approval. 
                    Attempting to login before approval will result in access denial.
                  </p>
                </div>
              </div>
            </motion.div>

            {/* Timeline */}
            <motion.div
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              transition={{ delay: 0.8 }}
              className="bg-gray-700/30 rounded-2xl p-6 mb-8 text-left"
            >
              <h3 className="text-white font-bold text-lg mb-6 text-center">Application Timeline</h3>
              <div className="space-y-4">
                <div className="flex items-start gap-4">
                  <div className="w-8 h-8 rounded-full bg-emerald-500 flex items-center justify-center flex-shrink-0">
                    <CheckCircle className="w-5 h-5 text-white" />
                  </div>
                  <div>
                    <p className="text-white font-semibold">Step 1: Application Submitted</p>
                    <p className="text-gray-400 text-sm">Your NDA request has been received ✅</p>
                  </div>
                </div>

                <div className="flex items-start gap-4">
                  <div className="w-8 h-8 rounded-full bg-purple-500 flex items-center justify-center flex-shrink-0">
                    <FileCheck className="w-5 h-5 text-white" />
                  </div>
                  <div>
                    <p className="text-white font-semibold">Step 2: Admin Review</p>
                    <p className="text-gray-400 text-sm">Your documents and KPI data are being verified (24-48 hours)</p>
                  </div>
                </div>

                <div className="flex items-start gap-4">
                  <div className="w-8 h-8 rounded-full bg-gray-600 flex items-center justify-center flex-shrink-0">
                    <Mail className="w-5 h-5 text-white" />
                  </div>
                  <div>
                    <p className="text-gray-300 font-semibold">Step 3: Approval Notification</p>
                    <p className="text-gray-400 text-sm">You'll receive login credentials via email</p>
                  </div>
                </div>

                <div className="flex items-start gap-4">
                  <div className="w-8 h-8 rounded-full bg-gray-600 flex items-center justify-center flex-shrink-0">
                    <CheckCircle className="w-5 h-5 text-white" />
                  </div>
                  <div>
                    <p className="text-gray-300 font-semibold">Step 4: Start Working</p>
                    <p className="text-gray-400 text-sm">Access your manager dashboard and start earning</p>
                  </div>
                </div>
              </div>
            </motion.div>

            {/* Actions */}
            <div className="flex flex-col sm:flex-row gap-4 justify-center">
              <Link
                to="/"
                className="flex items-center justify-center gap-2 px-8 py-4 rounded-xl bg-gradient-to-r from-purple-600 to-purple-700 hover:from-purple-700 hover:to-purple-800 text-white font-semibold transition-all shadow-lg"
              >
                <Home className="w-5 h-5" />
                Back to Home
              </Link>
            </div>

            {/* Contact Support */}
            <motion.p
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 1 }}
              className="text-gray-400 text-sm mt-8"
            >
              Questions? Contact us at <span className="text-purple-400 font-semibold">support@7starswin.com</span>
            </motion.p>
          </motion.div>
        </div>

        <LegalFooter />
      </div>
    </div>
  );
}