import { useState } from 'react';
import { useParams, useNavigate } from 'react-router';
import { DollarSign, Mail, CheckCircle, AlertCircle } from 'lucide-react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

export default function PayoutProcess() {
  const { id } = useParams();
  const navigate = useNavigate();
  const [amount, setAmount] = useState('6000');
  const [notes, setNotes] = useState('');
  const [showSuccess, setShowSuccess] = useState(false);

  const manager = {
    name: 'Michael Chen',
    email: 'michael.chen@example.com',
    currentSalary: 6000,
    lastPayment: 'Jan 31, 2026',
  };

  const handlePayout = () => {
    // Mock payout processing
    setShowSuccess(true);
    setTimeout(() => {
      navigate('/admin/dashboard');
    }, 3000);
  };

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="admin" />
      <div className="flex-1 ml-64">
        <DashboardHeader title="Process Payout" subtitle="Send payment to manager" />
        
        <div className="p-8">
          <div className="max-w-4xl mx-auto space-y-6">
            {/* Manager Info */}
            <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
              <div className="flex items-center gap-4">
                <div className="w-16 h-16 rounded-full bg-gradient-to-br from-purple-600 to-blue-600 flex items-center justify-center">
                  <span className="text-white font-bold text-xl">
                    {manager.name.split(' ').map(n => n[0]).join('')}
                  </span>
                </div>
                <div>
                  <h2 className="text-white text-xl font-bold">{manager.name}</h2>
                  <p className="text-gray-400">{manager.email}</p>
                </div>
              </div>
              <div className="grid md:grid-cols-3 gap-4 mt-6">
                <div className="p-4 bg-gray-900/50 rounded-xl">
                  <p className="text-gray-400 text-sm mb-1">Current Salary</p>
                  <p className="text-white text-xl font-bold">${manager.currentSalary.toLocaleString()}</p>
                </div>
                <div className="p-4 bg-gray-900/50 rounded-xl">
                  <p className="text-gray-400 text-sm mb-1">Last Payment</p>
                  <p className="text-white text-xl font-bold">{manager.lastPayment}</p>
                </div>
                <div className="p-4 bg-gray-900/50 rounded-xl">
                  <p className="text-gray-400 text-sm mb-1">Status</p>
                  <p className="text-emerald-400 text-xl font-bold">Active</p>
                </div>
              </div>
            </div>

            {/* Payout Form */}
            {!showSuccess ? (
              <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-8">
                <div className="flex items-center gap-3 mb-6">
                  <div className="w-12 h-12 rounded-lg bg-emerald-600/20 flex items-center justify-center">
                    <DollarSign className="w-6 h-6 text-emerald-400" />
                  </div>
                  <h3 className="text-white text-2xl font-bold">Payout Details</h3>
                </div>

                <div className="space-y-6">
                  <div>
                    <label className="block text-white mb-2 font-medium">Payment Amount (USD)</label>
                    <div className="relative">
                      <span className="absolute left-4 top-1/2 -translate-y-1/2 text-gray-400 text-lg">$</span>
                      <input
                        type="number"
                        value={amount}
                        onChange={(e) => setAmount(e.target.value)}
                        className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-8 pr-4 py-4 text-white text-2xl font-bold focus:border-emerald-500 focus:outline-none"
                        placeholder="0.00"
                      />
                    </div>
                  </div>

                  <div>
                    <label className="block text-white mb-2 font-medium">Payment Notes (Optional)</label>
                    <textarea
                      value={notes}
                      onChange={(e) => setNotes(e.target.value)}
                      className="w-full bg-gray-700/50 border border-gray-600 rounded-xl px-4 py-3 text-white focus:border-emerald-500 focus:outline-none h-32"
                      placeholder="Add any notes about this payment..."
                    />
                  </div>

                  <div className="bg-blue-500/10 border border-blue-500/30 rounded-xl p-4">
                    <h4 className="text-blue-300 font-semibold mb-2 flex items-center gap-2">
                      <Mail className="w-5 h-5" />
                      Email Notification Preview
                    </h4>
                    <div className="bg-gray-900/50 rounded-lg p-4 mt-3 text-sm">
                      <p className="text-gray-400 mb-2">To: {manager.email}</p>
                      <p className="text-gray-400 mb-2">Subject: Payment Processed - ${amount}</p>
                      <div className="border-t border-gray-700 pt-3 mt-3">
                        <p className="text-white mb-2">Dear {manager.name},</p>
                        <p className="text-gray-400 mb-2">
                          Your payment of ${amount} USD has been processed successfully and will be 
                          deposited to your account within 2-3 business days.
                        </p>
                        {notes && (
                          <div className="mt-3 p-3 bg-gray-800 rounded">
                            <p className="text-gray-400 text-xs mb-1">Notes:</p>
                            <p className="text-white">{notes}</p>
                          </div>
                        )}
                      </div>
                    </div>
                  </div>

                  <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-xl p-4">
                    <p className="text-yellow-300 text-sm flex items-center gap-2">
                      <AlertCircle className="w-4 h-4" />
                      Please verify all payment details before confirming. This action cannot be undone.
                    </p>
                  </div>

                  <button
                    onClick={handlePayout}
                    className="w-full px-6 py-4 rounded-xl bg-gradient-to-r from-emerald-600 to-emerald-700 hover:from-emerald-700 hover:to-emerald-800 text-white text-lg font-semibold shadow-lg shadow-emerald-600/50 transition-all"
                  >
                    Confirm Payout - ${amount}
                  </button>
                </div>
              </div>
            ) : (
              /* Success Modal */
              <div className="bg-gray-800/50 backdrop-blur-lg border border-emerald-600 rounded-2xl p-12 text-center">
                <div className="w-20 h-20 rounded-full bg-emerald-600/20 flex items-center justify-center mx-auto mb-6">
                  <CheckCircle className="w-12 h-12 text-emerald-400" />
                </div>
                <h2 className="text-white text-3xl font-bold mb-4">Payment Processed!</h2>
                <p className="text-gray-400 text-lg mb-2">
                  Successfully sent ${amount} to {manager.name}
                </p>
                <p className="text-gray-500 text-sm mb-6">
                  Email notification has been sent to {manager.email}
                </p>
                <div className="bg-emerald-500/10 border border-emerald-500/30 rounded-xl p-4 max-w-md mx-auto">
                  <p className="text-emerald-300 text-sm">
                    Funds will be deposited within 2-3 business days
                  </p>
                </div>
                <p className="text-gray-500 text-sm mt-6">Redirecting to dashboard...</p>
              </div>
            )}

            {/* Recent Payouts */}
            {!showSuccess && (
              <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
                <h3 className="text-white text-lg font-bold mb-4">Recent Payouts</h3>
                <div className="space-y-3">
                  {[
                    { date: 'Jan 31, 2026', amount: 6000, status: 'Completed' },
                    { date: 'Dec 31, 2025', amount: 6000, status: 'Completed' },
                    { date: 'Nov 30, 2025', amount: 5500, status: 'Completed' },
                  ].map((payout, index) => (
                    <div key={index} className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                      <div>
                        <p className="text-white font-medium">{payout.date}</p>
                        <p className="text-gray-400 text-sm">{payout.status}</p>
                      </div>
                      <p className="text-emerald-400 text-xl font-bold">${payout.amount.toLocaleString()}</p>
                    </div>
                  ))}
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}
