import { useState } from 'react';
import { Lock, Shield, Key, Mail, Bell, Eye, EyeOff, CheckCircle } from 'lucide-react';
import Sidebar from '../components/Sidebar';
import DashboardHeader from '../components/DashboardHeader';

export default function SettingsSecurity() {
  const [showCurrentPassword, setShowCurrentPassword] = useState(false);
  const [showNewPassword, setShowNewPassword] = useState(false);
  const [twoFactorEnabled, setTwoFactorEnabled] = useState(false);
  const [emailNotifications, setEmailNotifications] = useState(true);
  const [pushNotifications, setPushNotifications] = useState(true);

  const [passwordData, setPasswordData] = useState({
    currentPassword: '',
    newPassword: '',
    confirmPassword: '',
  });

  const handlePasswordChange = () => {
    if (passwordData.newPassword !== passwordData.confirmPassword) {
      alert('Passwords do not match!');
      return;
    }
    alert('Password changed successfully!');
    setPasswordData({ currentPassword: '', newPassword: '', confirmPassword: '' });
  };

  return (
    <div className="flex min-h-screen bg-gray-950">
      <Sidebar role="manager" />
      <div className="flex-1 ml-64">
        <DashboardHeader title="Settings & Security" subtitle="Manage your account security and preferences" />
        
        <div className="p-8 space-y-6">
          {/* Security Overview */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center gap-3 mb-6">
              <div className="w-12 h-12 rounded-lg bg-emerald-600/20 flex items-center justify-center">
                <Shield className="w-6 h-6 text-emerald-400" />
              </div>
              <div>
                <h2 className="text-white text-xl font-bold">Security Status</h2>
                <p className="text-gray-400 text-sm">Your account security level</p>
              </div>
            </div>
            <div className="grid md:grid-cols-3 gap-4">
              <div className="p-4 bg-emerald-500/10 border border-emerald-500/30 rounded-xl">
                <div className="flex items-center justify-between mb-2">
                  <span className="text-emerald-400 text-sm font-medium">Password</span>
                  <CheckCircle className="w-5 h-5 text-emerald-400" />
                </div>
                <p className="text-white font-semibold">Strong</p>
              </div>
              <div className="p-4 bg-yellow-500/10 border border-yellow-500/30 rounded-xl">
                <div className="flex items-center justify-between mb-2">
                  <span className="text-yellow-400 text-sm font-medium">2FA</span>
                  <span className="text-yellow-400 text-xs">Recommended</span>
                </div>
                <p className="text-white font-semibold">
                  {twoFactorEnabled ? 'Enabled' : 'Disabled'}
                </p>
              </div>
              <div className="p-4 bg-blue-500/10 border border-blue-500/30 rounded-xl">
                <div className="flex items-center justify-between mb-2">
                  <span className="text-blue-400 text-sm font-medium">Encryption</span>
                  <CheckCircle className="w-5 h-5 text-blue-400" />
                </div>
                <p className="text-white font-semibold">256-bit SSL</p>
              </div>
            </div>
          </div>

          {/* Change Password */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center gap-3 mb-6">
              <div className="w-10 h-10 rounded-lg bg-purple-600/20 flex items-center justify-center">
                <Key className="w-5 h-5 text-purple-400" />
              </div>
              <h3 className="text-white text-lg font-bold">Change Password</h3>
            </div>
            <div className="space-y-4 max-w-2xl">
              <div>
                <label className="block text-white mb-2 font-medium">Current Password</label>
                <div className="relative">
                  <Lock className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                  <input
                    type={showCurrentPassword ? 'text' : 'password'}
                    value={passwordData.currentPassword}
                    onChange={(e) => setPasswordData({ ...passwordData, currentPassword: e.target.value })}
                    className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-12 py-3 text-white focus:border-purple-500 focus:outline-none"
                    placeholder="Enter current password"
                  />
                  <button
                    type="button"
                    onClick={() => setShowCurrentPassword(!showCurrentPassword)}
                    className="absolute right-4 top-1/2 -translate-y-1/2 text-gray-400 hover:text-white"
                  >
                    {showCurrentPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                  </button>
                </div>
              </div>

              <div>
                <label className="block text-white mb-2 font-medium">New Password</label>
                <div className="relative">
                  <Lock className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                  <input
                    type={showNewPassword ? 'text' : 'password'}
                    value={passwordData.newPassword}
                    onChange={(e) => setPasswordData({ ...passwordData, newPassword: e.target.value })}
                    className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-12 py-3 text-white focus:border-purple-500 focus:outline-none"
                    placeholder="Enter new password"
                  />
                  <button
                    type="button"
                    onClick={() => setShowNewPassword(!showNewPassword)}
                    className="absolute right-4 top-1/2 -translate-y-1/2 text-gray-400 hover:text-white"
                  >
                    {showNewPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                  </button>
                </div>
              </div>

              <div>
                <label className="block text-white mb-2 font-medium">Confirm New Password</label>
                <div className="relative">
                  <Lock className="absolute left-4 top-1/2 -translate-y-1/2 w-5 h-5 text-gray-400" />
                  <input
                    type="password"
                    value={passwordData.confirmPassword}
                    onChange={(e) => setPasswordData({ ...passwordData, confirmPassword: e.target.value })}
                    className="w-full bg-gray-700/50 border border-gray-600 rounded-xl pl-12 pr-4 py-3 text-white focus:border-purple-500 focus:outline-none"
                    placeholder="Confirm new password"
                  />
                </div>
              </div>

              <div className="bg-blue-500/10 border border-blue-500/30 rounded-xl p-4">
                <p className="text-blue-300 text-sm">
                  💡 Password must be at least 8 characters long and include uppercase, lowercase, numbers, and special characters.
                </p>
              </div>

              <button
                onClick={handlePasswordChange}
                className="px-6 py-3 rounded-xl bg-gradient-to-r from-purple-600 to-purple-700 hover:from-purple-700 hover:to-purple-800 text-white font-semibold shadow-lg shadow-purple-600/50"
              >
                Update Password
              </button>
            </div>
          </div>

          {/* Two-Factor Authentication */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center justify-between mb-6">
              <div className="flex items-center gap-3">
                <div className="w-10 h-10 rounded-lg bg-blue-600/20 flex items-center justify-center">
                  <Shield className="w-5 h-5 text-blue-400" />
                </div>
                <div>
                  <h3 className="text-white text-lg font-bold">Two-Factor Authentication</h3>
                  <p className="text-gray-400 text-sm">Add an extra layer of security</p>
                </div>
              </div>
              <label className="relative inline-flex items-center cursor-pointer">
                <input
                  type="checkbox"
                  checked={twoFactorEnabled}
                  onChange={(e) => setTwoFactorEnabled(e.target.checked)}
                  className="sr-only peer"
                />
                <div className="w-14 h-7 bg-gray-700 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-purple-800 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-0.5 after:left-[4px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-6 after:w-6 after:transition-all peer-checked:bg-purple-600"></div>
              </label>
            </div>
            {twoFactorEnabled && (
              <div className="space-y-4 max-w-2xl">
                <div className="bg-purple-500/10 border border-purple-500/30 rounded-xl p-4">
                  <p className="text-purple-300 text-sm mb-3">
                    <strong>✓ 2FA is enabled</strong> - Your account is protected with two-factor authentication.
                  </p>
                  <button className="text-purple-400 hover:text-purple-300 text-sm font-medium">
                    View Recovery Codes
                  </button>
                </div>
              </div>
            )}
            {!twoFactorEnabled && (
              <div className="bg-yellow-500/10 border border-yellow-500/30 rounded-xl p-4 max-w-2xl">
                <p className="text-yellow-300 text-sm">
                  ⚠️ We recommend enabling 2FA to secure your account with an additional verification step during login.
                </p>
              </div>
            )}
          </div>

          {/* Notification Preferences */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center gap-3 mb-6">
              <div className="w-10 h-10 rounded-lg bg-amber-600/20 flex items-center justify-center">
                <Bell className="w-5 h-5 text-amber-400" />
              </div>
              <h3 className="text-white text-lg font-bold">Notification Preferences</h3>
            </div>
            <div className="space-y-4 max-w-2xl">
              <div className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                <div className="flex items-center gap-3">
                  <Mail className="w-5 h-5 text-gray-400" />
                  <div>
                    <p className="text-white font-medium">Email Notifications</p>
                    <p className="text-gray-400 text-sm">Receive updates via email</p>
                  </div>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input
                    type="checkbox"
                    checked={emailNotifications}
                    onChange={(e) => setEmailNotifications(e.target.checked)}
                    className="sr-only peer"
                  />
                  <div className="w-14 h-7 bg-gray-700 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-purple-800 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-0.5 after:left-[4px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-6 after:w-6 after:transition-all peer-checked:bg-purple-600"></div>
                </label>
              </div>

              <div className="flex items-center justify-between p-4 bg-gray-900/50 rounded-xl">
                <div className="flex items-center gap-3">
                  <Bell className="w-5 h-5 text-gray-400" />
                  <div>
                    <p className="text-white font-medium">Push Notifications</p>
                    <p className="text-gray-400 text-sm">Receive push notifications</p>
                  </div>
                </div>
                <label className="relative inline-flex items-center cursor-pointer">
                  <input
                    type="checkbox"
                    checked={pushNotifications}
                    onChange={(e) => setPushNotifications(e.target.checked)}
                    className="sr-only peer"
                  />
                  <div className="w-14 h-7 bg-gray-700 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-purple-800 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-0.5 after:left-[4px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-6 after:w-6 after:transition-all peer-checked:bg-purple-600"></div>
                </label>
              </div>
            </div>
          </div>

          {/* Data Encryption Status */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <div className="flex items-center gap-3 mb-6">
              <div className="w-10 h-10 rounded-lg bg-emerald-600/20 flex items-center justify-center">
                <Lock className="w-5 h-5 text-emerald-400" />
              </div>
              <h3 className="text-white text-lg font-bold">Data Encryption Status</h3>
            </div>
            <div className="space-y-4 max-w-2xl">
              <div className="p-4 bg-emerald-500/10 border border-emerald-500/30 rounded-xl">
                <div className="flex items-center gap-3 mb-3">
                  <CheckCircle className="w-6 h-6 text-emerald-400" />
                  <p className="text-emerald-400 font-semibold">All Data Encrypted</p>
                </div>
                <p className="text-gray-300 text-sm leading-relaxed">
                  Your personal information, documents, and communications are protected with 
                  industry-standard 256-bit SSL/TLS encryption. All data is encrypted both in 
                  transit and at rest.
                </p>
              </div>
              <div className="grid md:grid-cols-2 gap-4">
                <div className="p-4 bg-gray-900/50 rounded-xl">
                  <p className="text-gray-400 text-sm mb-1">Encryption Type</p>
                  <p className="text-white font-semibold">AES-256</p>
                </div>
                <div className="p-4 bg-gray-900/50 rounded-xl">
                  <p className="text-gray-400 text-sm mb-1">SSL Certificate</p>
                  <p className="text-white font-semibold">Valid</p>
                </div>
              </div>
            </div>
          </div>

          {/* Legal Information */}
          <div className="bg-gray-800/50 backdrop-blur-lg border border-gray-700 rounded-2xl p-6">
            <h3 className="text-white text-lg font-bold mb-6">Legal & Privacy</h3>
            <div className="grid md:grid-cols-2 gap-4">
              <a href="#" className="p-4 bg-gray-900/50 rounded-xl hover:bg-gray-900/70 transition-colors group">
                <p className="text-white font-medium mb-1 group-hover:text-purple-400">Privacy Policy</p>
                <p className="text-gray-400 text-sm">Read our privacy policy</p>
              </a>
              <a href="#" className="p-4 bg-gray-900/50 rounded-xl hover:bg-gray-900/70 transition-colors group">
                <p className="text-white font-medium mb-1 group-hover:text-purple-400">Terms of Service</p>
                <p className="text-gray-400 text-sm">View terms and conditions</p>
              </a>
              <a href="#" className="p-4 bg-gray-900/50 rounded-xl hover:bg-gray-900/70 transition-colors group">
                <p className="text-white font-medium mb-1 group-hover:text-purple-400">Cookie Policy</p>
                <p className="text-gray-400 text-sm">Manage cookie preferences</p>
              </a>
              <a href="#" className="p-4 bg-gray-900/50 rounded-xl hover:bg-gray-900/70 transition-colors group">
                <p className="text-white font-medium mb-1 group-hover:text-purple-400">Data Protection</p>
                <p className="text-gray-400 text-sm">GDPR compliance info</p>
              </a>
            </div>
          </div>

          {/* Confidentiality Statement */}
          <div className="bg-purple-500/10 border border-purple-500/30 rounded-xl p-6">
            <h4 className="text-purple-300 font-semibold mb-3 flex items-center gap-2">
              <Shield className="w-5 h-5" />
              Confidentiality Statement
            </h4>
            <p className="text-gray-300 text-sm leading-relaxed">
              All information stored and processed through this platform is treated with strict confidentiality. 
              We employ end-to-end encryption, secure data centers, and follow industry best practices to protect 
              your sensitive data. Your personal information, KPI data, and documents are accessible only to 
              authorized personnel and are never shared with third parties without your explicit consent.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
